-- StaySpot Unified Database Schema
-- Combined from all database files with comprehensive features
-- Version: 3.0.0
-- Database: stayspotco_stayspot

SET FOREIGN_KEY_CHECKS = 0;

-- Create database
CREATE DATABASE IF NOT EXISTS stayspotco_stayspot;
USE stayspotco_stayspot;

-- Drop existing tables in correct order
DROP TABLE IF EXISTS 
    vendor_reviews, vendor_documents, vendor_assignments, vendor_service_categories, vendors,
    inspection_items, inspections, emergency_contacts, property_analytics, mobile_app_tokens,
    property_favorites, rent_reminders, applications, expenses, audit_logs, user_sessions,
    auth_attempts, system_settings, chat_messages, partnership_applications,
    transportation_bookings, analytics_data, reviews, documents, messages,
    maintenance_photos, maintenance_notes, maintenance_requests, notifications,
    payments, leases, property_documents, property_images, properties,
    insurance_policies, leads, users,
    service_categories, payment_statuses, lease_statuses, maintenance_statuses,
    maintenance_priorities, property_statuses, property_types, user_roles,
    chat_rooms, drivers, transportation_vehicles;

SET FOREIGN_KEY_CHECKS = 1;

-- ==================== ENUMERATION TABLES ====================

CREATE TABLE user_roles (
    role_code VARCHAR(50) PRIMARY KEY,
    role_name VARCHAR(100) NOT NULL,
    description TEXT,
    permissions JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE property_types (
    type_code VARCHAR(50) PRIMARY KEY,
    type_name VARCHAR(100) NOT NULL,
    description TEXT,
    icon_class VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE property_statuses (
    status_code VARCHAR(50) PRIMARY KEY,
    status_name VARCHAR(100) NOT NULL,
    description TEXT,
    color_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE maintenance_priorities (
    priority_code VARCHAR(20) PRIMARY KEY,
    priority_name VARCHAR(50) NOT NULL,
    description TEXT,
    sla_hours INT,
    color_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE maintenance_statuses (
    status_code VARCHAR(50) PRIMARY KEY,
    status_name VARCHAR(100) NOT NULL,
    description TEXT,
    color_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE lease_statuses (
    status_code VARCHAR(50) PRIMARY KEY,
    status_name VARCHAR(100) NOT NULL,
    description TEXT,
    color_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE payment_statuses (
    status_code VARCHAR(50) PRIMARY KEY,
    status_name VARCHAR(100) NOT NULL,
    description TEXT,
    color_class VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
);

CREATE TABLE service_categories (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    category_code VARCHAR(50) NOT NULL UNIQUE,
    category_name VARCHAR(100) NOT NULL,
    description TEXT,
    parent_category_code VARCHAR(50) NULL,
    icon_class VARCHAR(100),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_parent_category (parent_category_code),
    FOREIGN KEY (parent_category_code) REFERENCES service_categories(category_code)
);

-- ==================== CORE USER TABLES ====================

CREATE TABLE users (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Basic Information
    first_name VARCHAR(50) NOT NULL,
    last_name VARCHAR(50) NOT NULL,
    email VARCHAR(255) NOT NULL UNIQUE,
    phone VARCHAR(20),
    password_hash VARCHAR(255) NOT NULL,
    
    -- Role Management
    role_code VARCHAR(50) NOT NULL,
    
    -- Profile Information
    avatar_url VARCHAR(500),
    profile_image VARCHAR(255),
    bio TEXT,
    date_of_birth DATE,
    
    -- Address Information
    address_street VARCHAR(255),
    address_city VARCHAR(100),
    address_state VARCHAR(100),
    address_zip_code VARCHAR(20),
    address_country VARCHAR(100) DEFAULT 'KE',
    latitude DECIMAL(10,8),
    longitude DECIMAL(11,8),
    
    -- Company/Professional Information
    company_name VARCHAR(255),
    company_position VARCHAR(100),
    license_number VARCHAR(100),
    tax_id VARCHAR(100),
    
    -- Emergency Contact
    emergency_contact_name VARCHAR(100),
    emergency_contact_phone VARCHAR(20),
    emergency_contact_relationship VARCHAR(50),
    
    -- Preferences & Settings
    preferences JSON,
    settings JSON,
    
    -- Security & Status
    is_email_verified BOOLEAN DEFAULT FALSE,
    is_phone_verified BOOLEAN DEFAULT FALSE,
    is_active BOOLEAN DEFAULT TRUE,
    last_login DATETIME,
    
    -- Two-Factor Authentication
    two_factor_enabled BOOLEAN DEFAULT FALSE,
    two_factor_secret VARCHAR(255),
    
    -- Password Management
    password_reset_token VARCHAR(255),
    password_reset_expires DATETIME,
    
    -- Soft Delete
    deleted_at DATETIME NULL,
    
    -- Audit Fields
    created_by BIGINT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_email (email),
    INDEX idx_role (role_code),
    INDEX idx_active (is_active),
    INDEX idx_company (company_name),
    INDEX idx_created_at (created_at),
    INDEX idx_deleted (deleted_at),
    INDEX idx_name (first_name, last_name),
    
    -- Foreign Keys
    FOREIGN KEY (role_code) REFERENCES user_roles(role_code),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

-- ==================== PROPERTY MANAGEMENT ====================

CREATE TABLE properties (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Basic Information
    title VARCHAR(255) NOT NULL,
    description TEXT,
    property_type_code VARCHAR(50) NOT NULL,
    status_code VARCHAR(50) NOT NULL,
    
    -- Location Details
    address_street VARCHAR(255) NOT NULL,
    address_city VARCHAR(100) NOT NULL,
    address_state VARCHAR(100) NOT NULL,
    address_zip_code VARCHAR(20) NOT NULL,
    address_country VARCHAR(100) DEFAULT 'KE',
    latitude DECIMAL(10,8),
    longitude DECIMAL(11,8),
    
    -- Property Specifications
    bedrooms INT NOT NULL DEFAULT 0,
    bathrooms DECIMAL(3,1) NOT NULL DEFAULT 0,
    area_sqft INT,
    square_feet INT,
    year_built YEAR,
    lot_size_sqft INT,
    
    -- Rental Information
    monthly_rent DECIMAL(10,2) NOT NULL,
    rent_amount DECIMAL(10,2) NOT NULL,
    security_deposit DECIMAL(10,2),
    deposit_amount DECIMAL(10,2),
    available_from DATE,
    is_published BOOLEAN DEFAULT FALSE,
    is_available BOOLEAN DEFAULT TRUE,
    
    -- Features & Amenities
    amenities JSON,
    interior_features JSON,
    kitchen_amenities JSON,
    bathroom_features JSON,
    outdoor_amenities JSON,
    smart_home_features JSON,
    accessibility_features JSON,
    utility_information JSON,
    rules JSON,
    features JSON,
    
    -- Media
    main_image_url VARCHAR(500),
    images JSON,
    
    -- Ownership & Management
    owner_id BIGINT NOT NULL,
    property_manager_id BIGINT NULL,
    primary_agent_id BIGINT NULL,
    
    -- Soft Delete
    deleted_at DATETIME NULL,
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_owner (owner_id),
    INDEX idx_manager (property_manager_id),
    INDEX idx_agent (primary_agent_id),
    INDEX idx_location (address_city, address_state),
    INDEX idx_rent (monthly_rent),
    INDEX idx_bedrooms (bedrooms),
    INDEX idx_status (status_code),
    INDEX idx_available (available_from),
    INDEX idx_deleted (deleted_at),
    INDEX idx_published (is_published),
    FULLTEXT idx_search (title, description, address_street),
    
    -- Foreign Keys
    FOREIGN KEY (property_type_code) REFERENCES property_types(type_code),
    FOREIGN KEY (status_code) REFERENCES property_statuses(status_code),
    FOREIGN KEY (owner_id) REFERENCES users(id),
    FOREIGN KEY (property_manager_id) REFERENCES users(id),
    FOREIGN KEY (primary_agent_id) REFERENCES users(id),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

CREATE TABLE property_images (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    property_id BIGINT NOT NULL,
    image_url VARCHAR(500) NOT NULL,
    image_order INT DEFAULT 0,
    caption VARCHAR(255),
    is_primary BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_property (property_id),
    INDEX idx_order (image_order),
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE
);

CREATE TABLE property_documents (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    property_id BIGINT NOT NULL,
    document_name VARCHAR(255) NOT NULL,
    document_url VARCHAR(500) NOT NULL,
    document_type ENUM(
        'lease_agreement',
        'id_proof',
        'income_proof',
        'property_document',
        'maintenance_report',
        'inspection_report',
        'insurance_document',
        'other'
    ) NOT NULL,
    category VARCHAR(100),
    file_size BIGINT,
    mime_type VARCHAR(100),
    tags JSON,
    is_public BOOLEAN DEFAULT FALSE,
    expires_at DATETIME,
    uploaded_by BIGINT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_property (property_id),
    INDEX idx_type (document_type),
    INDEX idx_uploaded_by (uploaded_by),
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (uploaded_by) REFERENCES users(id)
);

-- ==================== LEASE MANAGEMENT ====================

CREATE TABLE leases (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Lease Information
    property_id BIGINT NOT NULL,
    tenant_id BIGINT NOT NULL,
    landlord_id BIGINT NOT NULL,
    lease_number VARCHAR(100) NOT NULL UNIQUE,
    
    -- Term Details
    start_date DATE NOT NULL,
    end_date DATE NOT NULL,
    monthly_rent DECIMAL(10,2) NOT NULL,
    security_deposit DECIMAL(10,2) NOT NULL,
    
    -- Occupants
    primary_tenant_id BIGINT NOT NULL,
    occupants_count INT DEFAULT 1,
    occupants_details JSON,
    
    -- Lease Status
    status_code VARCHAR(50) NOT NULL DEFAULT 'pending',
    
    -- Documents
    lease_document_url VARCHAR(500),
    lease_terms TEXT,
    
    -- Financial
    rent_due_day INT DEFAULT 1,
    late_fee_percentage DECIMAL(5,2) DEFAULT 5.0,
    grace_period_days INT DEFAULT 5,
    
    -- Special Terms
    special_terms TEXT,
    special_conditions TEXT,
    terms JSON,
    
    -- Dates
    signed_at DATETIME,
    terminated_at DATETIME,
    
    -- Soft Delete
    deleted_at DATETIME NULL,
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_property (property_id),
    INDEX idx_tenant (tenant_id),
    INDEX idx_landlord (landlord_id),
    INDEX idx_primary_tenant (primary_tenant_id),
    INDEX idx_status (status_code),
    INDEX idx_dates (start_date, end_date),
    INDEX idx_lease_number (lease_number),
    INDEX idx_deleted (deleted_at),
    
    -- Foreign Keys
    FOREIGN KEY (property_id) REFERENCES properties(id),
    FOREIGN KEY (tenant_id) REFERENCES users(id),
    FOREIGN KEY (landlord_id) REFERENCES users(id),
    FOREIGN KEY (primary_tenant_id) REFERENCES users(id),
    FOREIGN KEY (status_code) REFERENCES lease_statuses(status_code),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

-- ==================== PAYMENT MANAGEMENT ====================

CREATE TABLE payments (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Payment Details
    lease_id BIGINT NOT NULL,
    tenant_id BIGINT NOT NULL,
    landlord_id BIGINT NOT NULL,
    property_id BIGINT NOT NULL,
    
    -- Amount Information
    amount DECIMAL(10,2) NOT NULL,
    amount_due DECIMAL(10,2) NOT NULL,
    amount_paid DECIMAL(10,2) DEFAULT 0,
    late_fee DECIMAL(10,2) DEFAULT 0,
    late_fee_amount DECIMAL(10,2) DEFAULT 0,
    
    -- Period Information
    period_month INT NOT NULL,
    period_year INT NOT NULL,
    due_date DATE NOT NULL,
    paid_date DATETIME,
    
    -- Payment Method
    payment_method ENUM('credit_card', 'debit_card', 'bank_transfer', 'cash', 'check', 'online', 'paypal', 'stripe', 'mobile_money'),
    payment_type ENUM('rent', 'deposit', 'fee', 'utility', 'maintenance', 'late_fee') NOT NULL,
    transaction_id VARCHAR(255),
    payment_gateway VARCHAR(100),
    receipt_url VARCHAR(500),
    
    -- Status
    status_code VARCHAR(50) NOT NULL DEFAULT 'pending',
    
    -- Description
    description TEXT,
    notes TEXT,
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_lease (lease_id),
    INDEX idx_tenant (tenant_id),
    INDEX idx_landlord (landlord_id),
    INDEX idx_property (property_id),
    INDEX idx_period (period_year, period_month),
    INDEX idx_due_date (due_date),
    INDEX idx_paid_date (paid_date),
    INDEX idx_status (status_code),
    INDEX idx_transaction (transaction_id),
    
    -- Foreign Keys
    FOREIGN KEY (lease_id) REFERENCES leases(id),
    FOREIGN KEY (tenant_id) REFERENCES users(id),
    FOREIGN KEY (landlord_id) REFERENCES users(id),
    FOREIGN KEY (property_id) REFERENCES properties(id),
    FOREIGN KEY (status_code) REFERENCES payment_statuses(status_code),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

CREATE TABLE expenses (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Expense Details
    property_id BIGINT NOT NULL,
    category ENUM('maintenance', 'utilities', 'insurance', 'taxes', 'management', 'marketing', 'legal', 'other') NOT NULL,
    description TEXT NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    
    -- Vendor Information
    vendor_name VARCHAR(255),
    vendor_contact VARCHAR(255),
    vendor VARCHAR(255),
    
    -- Timing
    expense_date DATE NOT NULL,
    due_date DATE,
    paid_date DATETIME,
    
    -- Recurring Expenses
    is_recurring BOOLEAN DEFAULT FALSE,
    recurring_frequency ENUM('monthly', 'quarterly', 'yearly'),
    recurrence_pattern VARCHAR(50),
    
    -- Documentation
    receipt_url VARCHAR(500),
    
    -- Status
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    approved_by BIGINT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_property (property_id),
    INDEX idx_category (category),
    INDEX idx_date (expense_date),
    INDEX idx_vendor (vendor_name),
    INDEX idx_status (status),
    
    -- Foreign Keys
    FOREIGN KEY (property_id) REFERENCES properties(id),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (approved_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);
-- ==================== MAINTENANCE MANAGEMENT ====================

CREATE TABLE maintenance_requests (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Request Details
    property_id BIGINT NOT NULL,
    unit_id VARCHAR(100),
    requested_by BIGINT NOT NULL,
    tenant_id BIGINT NOT NULL,
    assigned_to BIGINT NULL,
    landlord_id BIGINT NOT NULL,
    
    -- Issue Information
    title VARCHAR(255) NOT NULL,
    description TEXT NOT NULL,
    category ENUM('plumbing', 'electrical', 'hvac', 'appliance', 'structural', 'cosmetic', 'pest_control', 'cleaning', 'other') NOT NULL DEFAULT 'other',
    priority ENUM('low', 'medium', 'high', 'urgent', 'emergency') NOT NULL DEFAULT 'medium',
    priority_code VARCHAR(20) NOT NULL DEFAULT 'normal',
    
    -- Status & Tracking
    status ENUM('pending', 'in_progress', 'completed', 'cancelled') NOT NULL DEFAULT 'pending',
    status_code VARCHAR(50) NOT NULL DEFAULT 'submitted',
    estimated_cost DECIMAL(10,2),
    actual_cost DECIMAL(10,2),
    cost DECIMAL(10,2),
    rating INT,
    
    -- Media
    images JSON,
    
    -- Scheduling
    requested_date DATETIME DEFAULT CURRENT_TIMESTAMP,
    scheduled_date DATETIME,
    completed_date DATETIME,
    
    -- Location Access
    access_instructions TEXT,
    tenant_available BOOLEAN DEFAULT FALSE,
    notes TEXT,
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_property (property_id),
    INDEX idx_requested_by (requested_by),
    INDEX idx_tenant (tenant_id),
    INDEX idx_assigned_to (assigned_to),
    INDEX idx_landlord (landlord_id),
    INDEX idx_priority (priority_code),
    INDEX idx_status (status_code),
    INDEX idx_category (category),
    INDEX idx_requested_date (requested_date),
    INDEX idx_scheduled_date (scheduled_date),
    
    -- Foreign Keys
    FOREIGN KEY (property_id) REFERENCES properties(id),
    FOREIGN KEY (requested_by) REFERENCES users(id),
    FOREIGN KEY (tenant_id) REFERENCES users(id),
    FOREIGN KEY (assigned_to) REFERENCES users(id),
    FOREIGN KEY (landlord_id) REFERENCES users(id),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

CREATE TABLE maintenance_photos (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    maintenance_request_id BIGINT NOT NULL,
    photo_url VARCHAR(500) NOT NULL,
    caption VARCHAR(255),
    photo_type ENUM('before', 'after', 'during') DEFAULT 'during',
    uploaded_by BIGINT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_request (maintenance_request_id),
    FOREIGN KEY (maintenance_request_id) REFERENCES maintenance_requests(id) ON DELETE CASCADE,
    FOREIGN KEY (uploaded_by) REFERENCES users(id)
);

CREATE TABLE maintenance_notes (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    maintenance_request_id BIGINT NOT NULL,
    note_text TEXT NOT NULL,
    added_by BIGINT NOT NULL,
    is_internal BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_request (maintenance_request_id),
    FOREIGN KEY (maintenance_request_id) REFERENCES maintenance_requests(id) ON DELETE CASCADE,
    FOREIGN KEY (added_by) REFERENCES users(id)
);

-- ==================== NOTIFICATION & MESSAGING ====================

CREATE TABLE notifications (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Notification Details
    user_id BIGINT NOT NULL,
    title VARCHAR(255) NOT NULL,
    message TEXT NOT NULL,
    type ENUM(
        'info',
        'success',
        'warning',
        'error',
        'payment',
        'maintenance',
        'lease',
        'payment_reminder',
        'maintenance_update', 
        'lease_expiry',
        'new_message',
        'system_alert',
        'booking_confirmation',
        'payment_received',
        'maintenance_scheduled'
    ) NOT NULL,
    
    -- Context
    entity_type ENUM('property', 'lease', 'payment', 'maintenance', 'user'),
    entity_id BIGINT,
    action_url VARCHAR(500),
    metadata JSON,
    
    -- Status
    is_read BOOLEAN DEFAULT FALSE,
    read_at DATETIME,
    priority ENUM('low', 'medium', 'high') DEFAULT 'medium',
    
    -- Delivery
    send_email BOOLEAN DEFAULT FALSE,
    send_sms BOOLEAN DEFAULT FALSE,
    email_sent BOOLEAN DEFAULT FALSE,
    sms_sent BOOLEAN DEFAULT FALSE,
    
    -- Expiry
    expires_at DATETIME,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_user (user_id),
    INDEX idx_type (type),
    INDEX idx_entity (entity_type, entity_id),
    INDEX idx_created (created_at),
    INDEX idx_read (is_read),
    INDEX idx_priority (priority),
    
    -- Foreign Keys
    FOREIGN KEY (user_id) REFERENCES users(id)
);

CREATE TABLE messages (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Message Details
    sender_id BIGINT NOT NULL,
    receiver_id BIGINT NOT NULL,
    subject VARCHAR(255),
    content TEXT NOT NULL,
    message_text TEXT NOT NULL,
    message_type ENUM('text', 'image', 'file', 'system') DEFAULT 'text',
    
    -- Context
    property_id BIGINT,
    maintenance_request_id BIGINT,
    lease_id BIGINT,
    parent_message_id BIGINT,
    conversation_id VARCHAR(100) NOT NULL,
    
    -- Attachments
    attachments JSON,
    file_url VARCHAR(255),
    
    -- Status
    is_read BOOLEAN DEFAULT FALSE,
    read_at DATETIME,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_sender (sender_id),
    INDEX idx_receiver (receiver_id),
    INDEX idx_property (property_id),
    INDEX idx_maintenance (maintenance_request_id),
    INDEX idx_lease (lease_id),
    INDEX idx_conversation (conversation_id),
    INDEX idx_created (created_at),
    
    -- Foreign Keys
    FOREIGN KEY (sender_id) REFERENCES users(id),
    FOREIGN KEY (receiver_id) REFERENCES users(id),
    FOREIGN KEY (property_id) REFERENCES properties(id),
    FOREIGN KEY (maintenance_request_id) REFERENCES maintenance_requests(id),
    FOREIGN KEY (lease_id) REFERENCES leases(id),
    FOREIGN KEY (parent_message_id) REFERENCES messages(id) ON DELETE CASCADE
);

-- ==================== CHAT SYSTEM ====================

CREATE TABLE chat_rooms (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    participant_1_id BIGINT NOT NULL,
    participant_2_id BIGINT NOT NULL,
    room_type ENUM('direct', 'group', 'support') DEFAULT 'direct',
    room_name VARCHAR(255),
    is_active BOOLEAN DEFAULT TRUE,
    last_message_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_participants (participant_1_id, participant_2_id),
    INDEX idx_active (is_active),
    FOREIGN KEY (participant_1_id) REFERENCES users(id),
    FOREIGN KEY (participant_2_id) REFERENCES users(id)
);

CREATE TABLE chat_messages (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    room_id BIGINT NOT NULL,
    sender_id BIGINT NOT NULL,
    message_text TEXT NOT NULL,
    message_type ENUM('text', 'image', 'file') DEFAULT 'text',
    file_url VARCHAR(255),
    is_read BOOLEAN DEFAULT FALSE,
    read_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_room (room_id),
    INDEX idx_sender (sender_id),
    INDEX idx_created (created_at),
    FOREIGN KEY (room_id) REFERENCES chat_rooms(id) ON DELETE CASCADE,
    FOREIGN KEY (sender_id) REFERENCES users(id) ON DELETE CASCADE
);

-- ==================== TRANSPORTATION SYSTEM ====================

CREATE TABLE drivers (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    user_id BIGINT NOT NULL UNIQUE,
    license_number VARCHAR(100) NOT NULL,
    license_expiry DATE NOT NULL,
    vehicle_registration VARCHAR(50),
    is_verified BOOLEAN DEFAULT FALSE,
    is_online BOOLEAN DEFAULT FALSE,
    is_available BOOLEAN DEFAULT TRUE,
    total_trips INT DEFAULT 0,
    completed_trips INT DEFAULT 0,
    average_rating DECIMAL(3,2) DEFAULT 0.00,
    total_earnings DECIMAL(15,2) DEFAULT 0.00,
    current_latitude DECIMAL(10,8),
    current_longitude DECIMAL(11,8),
    last_location_update DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_user (user_id),
    INDEX idx_verified (is_verified),
    INDEX idx_available (is_available),
    INDEX idx_location (current_latitude, current_longitude),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

CREATE TABLE transportation_vehicles (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    driver_id BIGINT NOT NULL,
    vehicle_type ENUM('truck', 'van', 'pickup', 'motorcycle', 'pickup_truck') NOT NULL,
    make VARCHAR(100),
    model VARCHAR(100),
    year YEAR,
    color VARCHAR(50),
    license_plate VARCHAR(20) NOT NULL,
    capacity_weight DECIMAL(8,2),
    capacity_volume DECIMAL(8,2),
    is_active BOOLEAN DEFAULT TRUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_driver (driver_id),
    INDEX idx_type (vehicle_type),
    INDEX idx_active (is_active),
    FOREIGN KEY (driver_id) REFERENCES drivers(id) ON DELETE CASCADE
);

CREATE TABLE transportation_bookings (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    user_id BIGINT NOT NULL,
    driver_id BIGINT NULL,
    service_type ENUM('moving', 'delivery', 'pickup') NOT NULL,
    pickup_address TEXT NOT NULL,
    delivery_address TEXT NOT NULL,
    pickup_date DATE NOT NULL,
    pickup_time TIME NOT NULL,
    estimated_cost DECIMAL(10,2),
    actual_cost DECIMAL(10,2),
    status ENUM('pending', 'confirmed', 'in_progress', 'completed', 'cancelled') DEFAULT 'pending',
    vehicle_type ENUM('truck', 'van', 'pickup', 'motorcycle') NOT NULL,
    items_description TEXT,
    special_instructions TEXT,
    tracking_code VARCHAR(50) UNIQUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_user (user_id),
    INDEX idx_driver (driver_id),
    INDEX idx_status (status),
    INDEX idx_tracking (tracking_code),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (driver_id) REFERENCES users(id) ON DELETE SET NULL
);

-- ==================== PARTNERSHIP SYSTEM ====================

CREATE TABLE partnership_applications (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    company_name VARCHAR(255) NOT NULL,
    contact_person VARCHAR(255) NOT NULL,
    email VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    partnership_type ENUM('vendor', 'service_provider', 'technology', 'marketing', 'financial') NOT NULL,
    description TEXT NOT NULL,
    website VARCHAR(255),
    company_references TEXT,
    status ENUM('pending', 'approved', 'rejected', 'under_review') DEFAULT 'pending',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_status (status),
    INDEX idx_type (partnership_type)
);

-- ==================== REVIEWS & RATINGS ====================

CREATE TABLE reviews (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Review Details
    rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    title VARCHAR(255),
    comment TEXT,
    review_type ENUM('property', 'landlord', 'tenant') NOT NULL,
    
    -- Response
    response TEXT,
    responded_at DATETIME,
    
    -- Status
    is_published BOOLEAN DEFAULT TRUE,
    helpful_count INT DEFAULT 0,
    
    -- Relationships
    reviewer_id BIGINT NOT NULL,
    reviewee_id BIGINT,
    property_id BIGINT,
    lease_id BIGINT,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_reviewer (reviewer_id),
    INDEX idx_reviewee (reviewee_id),
    INDEX idx_property (property_id),
    INDEX idx_lease (lease_id),
    INDEX idx_review_type (review_type),
    INDEX idx_rating (rating),
    INDEX idx_published (is_published),
    
    -- Foreign Keys
    FOREIGN KEY (reviewer_id) REFERENCES users(id),
    FOREIGN KEY (reviewee_id) REFERENCES users(id),
    FOREIGN KEY (property_id) REFERENCES properties(id),
    FOREIGN KEY (lease_id) REFERENCES leases(id)
);

-- ==================== DOCUMENTS MANAGEMENT ====================

CREATE TABLE documents (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    title VARCHAR(255) NOT NULL,
    description TEXT,
    file_url VARCHAR(500) NOT NULL,
    file_name VARCHAR(255) NOT NULL,
    file_size BIGINT,
    mime_type VARCHAR(100),
    document_type ENUM(
        'lease_agreement',
        'id_proof',
        'income_proof',
        'property_document',
        'maintenance_report',
        'inspection_report',
        'insurance_document',
        'other'
    ) NOT NULL,
    category VARCHAR(100),
    tags JSON,
    is_public BOOLEAN DEFAULT FALSE,
    expires_at DATE,
    uploaded_by BIGINT NOT NULL,
    property_id BIGINT,
    lease_id BIGINT,
    user_id BIGINT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_document_type (document_type),
    INDEX idx_property (property_id),
    INDEX idx_lease (lease_id),
    INDEX idx_uploaded_by (uploaded_by),
    FOREIGN KEY (uploaded_by) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (lease_id) REFERENCES leases(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

-- ==================== VENDOR MANAGEMENT ====================

CREATE TABLE vendors (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Vendor Information
    company_name VARCHAR(255) NOT NULL,
    contact_name VARCHAR(100),
    contact_person VARCHAR(255) NOT NULL,
    email VARCHAR(255),
    phone VARCHAR(20),
    address TEXT,
    
    -- Services
    service_categories JSON,
    service_category ENUM('plumbing', 'electrical', 'hvac', 'cleaning', 'landscaping', 'security', 'general_maintenance', 'other') NOT NULL,
    primary_service_category VARCHAR(100),
    description TEXT,
    
    -- Address
    address_street VARCHAR(255),
    address_city VARCHAR(100),
    address_state VARCHAR(100),
    address_zip_code VARCHAR(20),
    address_country VARCHAR(100) DEFAULT 'KE',
    
    -- Business Information
    tax_id VARCHAR(100),
    license_number VARCHAR(100),
    license_expiry DATE,
    insurance_expiry DATE,
    insurance_provider VARCHAR(255),
    insurance_policy_number VARCHAR(100),
    insurance_coverage_amount DECIMAL(15,2),
    insurance_info JSON,
    
    -- Rating & Performance
    rating DECIMAL(3,2) DEFAULT 0.00,
    average_rating DECIMAL(3,2) DEFAULT 0,
    total_jobs INT DEFAULT 0,
    completed_jobs INT DEFAULT 0,
    total_earnings DECIMAL(15,2) DEFAULT 0,
    
    -- Response Time Metrics
    avg_response_time_hours DECIMAL(5,2) DEFAULT 0,
    avg_completion_time_hours DECIMAL(5,2) DEFAULT 0,
    
    -- Status
    is_active BOOLEAN DEFAULT TRUE,
    is_approved BOOLEAN DEFAULT FALSE,
    is_verified BOOLEAN DEFAULT FALSE,
    verification_date DATETIME,
    
    -- Availability
    is_available BOOLEAN DEFAULT TRUE,
    max_jobs_per_week INT DEFAULT 5,
    
    -- Service Areas
    service_areas JSON,
    service_radius_miles INT DEFAULT 25,
    
    -- Hourly Rate
    hourly_rate DECIMAL(10,2),
    emergency_hourly_rate DECIMAL(10,2),
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_company (company_name),
    INDEX idx_category (service_category),
    INDEX idx_primary_category (primary_service_category),
    INDEX idx_approved (is_approved),
    INDEX idx_verified (is_verified),
    INDEX idx_available (is_available),
    INDEX idx_active (is_active),
    INDEX idx_rating (average_rating),
    INDEX idx_location (address_city, address_state),
    
    -- Foreign Keys
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

-- ==================== INSPECTIONS ====================

CREATE TABLE inspections (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Inspection Details
    property_id BIGINT NOT NULL,
    inspector_id BIGINT NOT NULL,
    inspection_type ENUM('move_in', 'move_out', 'routine', 'maintenance', 'safety') NOT NULL,
    
    -- Scheduling
    scheduled_date DATE NOT NULL,
    completed_date DATE,
    
    -- Results
    overall_score INT,
    overall_condition ENUM('excellent', 'good', 'fair', 'poor'),
    status ENUM('scheduled', 'in_progress', 'completed', 'cancelled') DEFAULT 'scheduled',
    
    -- Documentation
    report_url VARCHAR(500),
    checklist JSON,
    findings TEXT,
    recommendations TEXT,
    photos JSON,
    notes TEXT,
    
    -- Audit Fields
    created_by BIGINT NOT NULL,
    updated_by BIGINT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_property (property_id),
    INDEX idx_inspector (inspector_id),
    INDEX idx_scheduled (scheduled_date),
    INDEX idx_type (inspection_type),
    INDEX idx_status (status),
    
    -- Foreign Keys
    FOREIGN KEY (property_id) REFERENCES properties(id),
    FOREIGN KEY (inspector_id) REFERENCES users(id),
    FOREIGN KEY (created_by) REFERENCES users(id),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);

-- ==================== APPLICATIONS ====================

CREATE TABLE applications (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    property_id BIGINT NOT NULL,
    applicant_id BIGINT NOT NULL,
    application_type ENUM('rental', 'lease_renewal', 'transfer') DEFAULT 'rental',
    status ENUM('pending', 'approved', 'rejected', 'withdrawn') DEFAULT 'pending',
    move_in_date DATE,
    employment_info JSON,
    company_references JSON,
    background_check_status ENUM('pending', 'passed', 'failed'),
    credit_score INT,
    monthly_income DECIMAL(10,2),
    notes TEXT,
    reviewed_by BIGINT,
    reviewed_at TIMESTAMP,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_property (property_id),
    INDEX idx_applicant (applicant_id),
    INDEX idx_status (status),
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (applicant_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewed_by) REFERENCES users(id) ON DELETE SET NULL
);

-- ==================== ANALYTICS ====================

CREATE TABLE analytics_data (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    metric_name VARCHAR(100) NOT NULL,
    metric_value DECIMAL(15,2) NOT NULL,
    metric_type ENUM('revenue', 'occupancy', 'maintenance_cost', 'user_count', 'property_count') NOT NULL,
    period_type ENUM('daily', 'weekly', 'monthly', 'yearly') NOT NULL,
    period_date DATE NOT NULL,
    metadata JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_metric (metric_name),
    INDEX idx_type (metric_type),
    INDEX idx_period (period_date)
);

CREATE TABLE property_analytics (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    property_id BIGINT NOT NULL,
    metric_date DATE NOT NULL,
    views_count INT DEFAULT 0,
    inquiries_count INT DEFAULT 0,
    applications_count INT DEFAULT 0,
    occupancy_rate DECIMAL(5,2) DEFAULT 0,
    revenue DECIMAL(10,2) DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    UNIQUE KEY unique_property_date (property_id, metric_date),
    INDEX idx_property (property_id),
    INDEX idx_date (metric_date),
    FOREIGN KEY (property_id) REFERENCES properties(id)
);

-- ==================== ADDITIONAL FEATURES ====================

CREATE TABLE emergency_contacts (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    property_id BIGINT NOT NULL,
    contact_type ENUM('police', 'hospital', 'fire', 'security', 'utility', 'maintenance') NOT NULL,
    contact_name VARCHAR(255) NOT NULL,
    phone_number VARCHAR(20) NOT NULL,
    address TEXT,
    is_24_7 BOOLEAN DEFAULT FALSE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_property (property_id),
    INDEX idx_contact_type (contact_type),
    FOREIGN KEY (property_id) REFERENCES properties(id)
);

CREATE TABLE mobile_app_tokens (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    device_token VARCHAR(255) NOT NULL,
    device_type ENUM('ios', 'android') NOT NULL,
    app_version VARCHAR(20),
    is_active BOOLEAN DEFAULT TRUE,
    last_used DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_user (user_id),
    INDEX idx_token (device_token),
    INDEX idx_active (is_active),
    FOREIGN KEY (user_id) REFERENCES users(id)
);

CREATE TABLE property_favorites (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    property_id BIGINT NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    UNIQUE KEY unique_user_property (user_id, property_id),
    INDEX idx_user (user_id),
    INDEX idx_property (property_id),
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (property_id) REFERENCES properties(id)
);

CREATE TABLE rent_reminders (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    lease_id BIGINT NOT NULL,
    reminder_date DATE NOT NULL,
    reminder_type ENUM('7_days', '3_days', '1_day', 'due_date', 'overdue') NOT NULL,
    is_sent BOOLEAN DEFAULT FALSE,
    sent_at DATETIME,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_lease (lease_id),
    INDEX idx_reminder_date (reminder_date),
    INDEX idx_sent (is_sent),
    FOREIGN KEY (lease_id) REFERENCES leases(id)
);

-- ==================== SECURITY & AUDIT ====================

CREATE TABLE audit_logs (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    
    -- Audit Details
    user_id BIGINT,
    action VARCHAR(100) NOT NULL,
    table_name VARCHAR(100) NOT NULL,
    entity_type ENUM('user', 'property', 'lease', 'payment', 'maintenance', 'document', 'review', 'vendor') NOT NULL,
    entity_id BIGINT NOT NULL,
    record_id BIGINT NOT NULL,
    
    -- Changes
    old_values JSON,
    new_values JSON,
    
    -- Context
    ip_address VARCHAR(45),
    user_agent TEXT,
    description TEXT,
    affected_user_id BIGINT,
    
    -- Timestamp
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_user (user_id),
    INDEX idx_entity (entity_type, entity_id),
    INDEX idx_action (action),
    INDEX idx_table (table_name),
    INDEX idx_created (created_at),
    
    -- Foreign Keys
    FOREIGN KEY (user_id) REFERENCES users(id),
    FOREIGN KEY (affected_user_id) REFERENCES users(id)
);

CREATE TABLE auth_attempts (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    email VARCHAR(255) NOT NULL,
    ip_address VARCHAR(45) NOT NULL,
    attempt_date TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    successful BOOLEAN DEFAULT FALSE,
    
    INDEX idx_email_date (email, attempt_date),
    INDEX idx_ip_date (ip_address, attempt_date)
);

CREATE TABLE user_sessions (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    user_id BIGINT NOT NULL,
    session_token VARCHAR(255) NOT NULL UNIQUE,
    expires_at DATETIME NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    INDEX idx_token (session_token),
    INDEX idx_user (user_id),
    INDEX idx_expires (expires_at),
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE
);

CREATE TABLE system_settings (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(255) NOT NULL UNIQUE,
    setting_value TEXT,
    setting_type ENUM('string', 'number', 'boolean', 'json') DEFAULT 'string',
    description TEXT,
    is_public BOOLEAN DEFAULT FALSE,
    updated_by BIGINT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    INDEX idx_key (setting_key),
    FOREIGN KEY (updated_by) REFERENCES users(id)
);
-- ==================== INSERT DEFAULT DATA ====================

-- Insert User Roles
INSERT INTO user_roles (role_code, role_name, description) VALUES
('super_admin', 'Super Administrator', 'Full platform access and system management'),
('system_admin', 'System Administrator', 'Enhanced system administrator with user management'),
('landlord', 'Property Owner/Landlord', 'Property owners and investors'),
('tenant', 'Tenant/Resident', 'Property residents and tenants'),
('property_manager', 'Property Manager', 'Professional property management'),
('property_agent', 'Property Agent', 'Real estate agents and brokers'),
('regional_manager', 'Regional Manager', 'Multi-location property management'),
('leasing_consultant', 'Leasing Consultant', 'Tenant acquisition specialists'),
('maintenance_supervisor', 'Maintenance Supervisor', 'Maintenance team leadership'),
('maintenance_staff', 'Maintenance Staff', 'Maintenance and repair technicians'),
('financial_officer', 'Financial Officer', 'Financial management and oversight'),
('financial_analyst', 'Financial Analyst', 'Financial analysis and reporting'),
('marketing_specialist', 'Marketing Specialist', 'Property marketing and advertising'),
('legal_advisor', 'Legal Advisor', 'Legal compliance and advice'),
('insurance_coordinator', 'Insurance Coordinator', 'Insurance policy management'),
('relocation_specialist', 'Relocation Specialist', 'Tenant relocation services'),
('community_manager', 'Community Manager', 'Apartment community management'),
('inspector', 'Inspector/Quality Assurance', 'Property inspections and quality control'),
('vendor', 'Vendor/Contractor', 'External service providers'),
('bnb_host', 'BNB Host', 'Short-term rental hosts'),
('corporate_housing_manager', 'Corporate Housing Manager', 'Corporate housing management'),
('student_housing_coordinator', 'Student Housing Coordinator', 'Student housing specialists'),
('luxury_property_specialist', 'Luxury Property Specialist', 'High-end property management'),
('data_analyst', 'Data Analyst', 'Business intelligence and analytics'),
('customer_support_agent', 'Customer Support Agent', 'User support and assistance'),
('driver', 'Driver', 'Transportation and moving services');

-- Insert Property Types
INSERT INTO property_types (type_code, type_name, description) VALUES
('apartment', 'Apartment', 'Multi-unit residential building'),
('house', 'Single-Family Home', 'Detached single-family residence'),
('condo', 'Condominium', 'Individually owned unit in a complex'),
('townhouse', 'Townhouse', 'Multi-floor attached home'),
('studio', 'Studio', 'Single-room apartment'),
('villa', 'Villa', 'Luxury detached residence'),
('multi_family', 'Multi-Family', 'Building with multiple residential units'),
('commercial', 'Commercial Space', 'Business and commercial properties'),
('office', 'Office Space', 'Commercial office properties'),
('warehouse', 'Warehouse', 'Industrial storage facilities'),
('other', 'Other', 'Other property types');

-- Insert Property Statuses
INSERT INTO property_statuses (status_code, status_name, description) VALUES
('draft', 'Draft', 'Property listing not published'),
('available', 'Available', 'Property available for rent'),
('occupied', 'Occupied', 'Property currently rented'),
('maintenance', 'Under Maintenance', 'Property undergoing maintenance'),
('unavailable', 'Unavailable', 'Property not available for rent');

-- Insert Maintenance Priorities
INSERT INTO maintenance_priorities (priority_code, priority_name, description, sla_hours) VALUES
('emergency', 'Emergency', 'Immediate attention required - safety hazards', 2),
('urgent', 'Urgent', 'Critical issues affecting livability', 24),
('high', 'High', 'Important but not critical issues', 72),
('normal', 'Normal', 'Routine maintenance requests', 168),
('low', 'Low', 'Cosmetic or minor issues', 336);

-- Insert Maintenance Statuses
INSERT INTO maintenance_statuses (status_code, status_name, description) VALUES
('submitted', 'Submitted', 'Request submitted by tenant'),
('assigned', 'Assigned', 'Assigned to maintenance staff'),
('in_progress', 'In Progress', 'Work in progress'),
('completed', 'Completed', 'Work completed successfully'),
('cancelled', 'Cancelled', 'Request cancelled'),
('on_hold', 'On Hold', 'Work temporarily on hold');

-- Insert Lease Statuses
INSERT INTO lease_statuses (status_code, status_name, description) VALUES
('draft', 'Draft', 'Lease agreement in draft stage'),
('pending', 'Pending', 'Awaiting signature or approval'),
('active', 'Active', 'Lease currently in effect'),
('expired', 'Expired', 'Lease term completed'),
('terminated', 'Terminated', 'Lease ended prematurely'),
('renewed', 'Renewed', 'Lease extended for new term');

-- Insert Payment Statuses
INSERT INTO payment_statuses (status_code, status_name, description) VALUES
('pending', 'Pending', 'Payment initiated but not completed'),
('paid', 'Paid', 'Payment successfully completed'),
('failed', 'Failed', 'Payment failed or declined'),
('overdue', 'Overdue', 'Payment past due date'),
('partial', 'Partial', 'Partial payment received'),
('refunded', 'Refunded', 'Payment refunded to tenant'),
('completed', 'Completed', 'Payment successfully processed');

-- Insert Service Categories
INSERT INTO service_categories (category_code, category_name, description) VALUES
('plumbing_emergency', 'Emergency Plumbing', '24/7 emergency plumbing services'),
('plumbing_repair', 'Plumbing Repair', 'General plumbing repairs and maintenance'),
('plumbing_installation', 'Plumbing Installation', 'New plumbing fixture installation'),
('electrical_emergency', 'Emergency Electrical', '24/7 emergency electrical services'),
('electrical_repair', 'Electrical Repair', 'General electrical repairs'),
('electrical_installation', 'Electrical Installation', 'New electrical fixture installation'),
('hvac_repair', 'HVAC Repair', 'Heating and cooling system repairs'),
('hvac_maintenance', 'HVAC Maintenance', 'Regular HVAC maintenance'),
('hvac_installation', 'HVAC Installation', 'New HVAC system installation'),
('carpentry', 'Carpentry', 'Woodwork and structural repairs'),
('painting', 'Painting', 'Interior and exterior painting'),
('drywall', 'Drywall Repair', 'Drywall installation and repair'),
('appliance_repair', 'Appliance Repair', 'Home appliance repairs'),
('appliance_installation', 'Appliance Installation', 'New appliance installation'),
('landscaping', 'Landscaping', 'Yard maintenance and landscaping'),
('roofing', 'Roofing', 'Roof repairs and maintenance'),
('gutter_cleaning', 'Gutter Cleaning', 'Gutter cleaning and repair'),
('deep_cleaning', 'Deep Cleaning', 'Thorough cleaning services'),
('move_in_cleaning', 'Move-In/Move-Out Cleaning', 'Cleaning for tenant transitions'),
('carpet_cleaning', 'Carpet Cleaning', 'Professional carpet cleaning'),
('pest_control', 'Pest Control', 'Pest extermination and prevention'),
('emergency_boardup', 'Emergency Board-Up', 'Property securing after damage'),
('water_restoration', 'Water Damage Restoration', 'Water damage cleanup and restoration'),
('property_inspection', 'Property Inspection', 'Professional property inspections'),
('safety_inspection', 'Safety Inspection', 'Safety compliance inspections'),
('security', 'Security Services', 'Property security and monitoring'),
('general_maintenance', 'General Maintenance', 'General property maintenance'),
('other', 'Other Services', 'Other specialized services');

-- Insert System Settings
INSERT INTO system_settings (setting_key, setting_value, setting_type, description, is_public) VALUES
('company_name', 'StaySpot', 'string', 'Platform company name', TRUE),
('company_location', 'Nakuru, Kenya', 'string', 'Company headquarters location', TRUE),
('company_phone', '+254748041595', 'string', 'Company contact phone', TRUE),
('company_email', 'info@stayspot.co.ke', 'string', 'Company email address', TRUE),
('company_address', 'Nakuru, Kenya', 'string', 'Company address', TRUE),
('platform_version', '3.0.0', 'string', 'Current platform version', TRUE),
('service_areas', 'Nairobi, Mombasa, Kisumu, Nakuru, Eldoret, Thika, Machakos, Meru, Nyeri, Kakamega, Kitale, Garissa, Malindi, Lamu, Naivasha, Nanyuki', 'string', 'Cities and towns where StaySpot operates', TRUE),
('default_currency', 'KES', 'string', 'Default currency for payments', TRUE),
('rent_due_day', '1', 'number', 'Default day of month rent is due', FALSE),
('late_fee_percentage', '5.0', 'number', 'Default late fee percentage', FALSE),
('grace_period_days', '5', 'number', 'Default grace period for late payments', FALSE),
('maintenance_emergency_hours', '2', 'number', 'SLA for emergency maintenance', FALSE),
('maintenance_urgent_hours', '24', 'number', 'SLA for urgent maintenance', FALSE),
('auto_archive_days', '90', 'number', 'Days before archiving old records', FALSE),
('chat_message_retention_days', '365', 'number', 'Days to retain chat messages', FALSE),
('driver_location_update_interval', '30', 'number', 'Seconds between location updates', FALSE),
('emergency_hotline', '999', 'string', 'National emergency hotline', TRUE),
('support_hours', '24/7', 'string', 'Customer support availability', TRUE),
('mobile_app_version', '3.0.0', 'string', 'Current mobile app version', TRUE),
('push_notifications_enabled', 'true', 'boolean', 'Enable push notifications', FALSE),
('analytics_retention_days', '365', 'number', 'Days to retain analytics data', FALSE),
('app_name', 'StaySpot', 'string', 'Application name', TRUE),
('app_version', '3.0.0', 'string', 'Application version', TRUE),
('maintenance_mode', 'false', 'boolean', 'Enable maintenance mode', FALSE),
('max_file_size', '10485760', 'number', 'Maximum file upload size in bytes', FALSE);

-- ==================== CREATE SAMPLE DATA ====================

-- Note: Passwords are hashed version of 'password123'
INSERT INTO users (email, password_hash, first_name, last_name, phone, role_code, is_email_verified, is_active, company_name, address_country, created_at) VALUES
-- Administration
('admin@stayspot.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'Super', 'Admin', '+254748041595', 'super_admin', TRUE, TRUE, 'StaySpot Kenya', 'KE', NOW()),

-- Property Owners
('landlord@example.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'John', 'Kamau', '+254712345678', 'landlord', TRUE, TRUE, 'Kamau Properties', 'KE', NOW()),

-- Property Manager
('manager@stayspot.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'Grace', 'Wanjiku', '+254723456789', 'property_manager', TRUE, TRUE, 'StaySpot Management', 'KE', NOW()),

-- Tenants
('tenant1@example.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'Mary', 'Njeri', '+254734567890', 'tenant', TRUE, TRUE, NULL, 'KE', NOW()),
('tenant2@example.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'Peter', 'Mwangi', '+254745678901', 'tenant', TRUE, TRUE, NULL, 'KE', NOW()),

-- Driver
('driver@stayspot.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'James', 'Kiprotich', '+254756789012', 'driver', TRUE, TRUE, 'StaySpot Transport', 'KE', NOW());

-- Create sample properties across major Kenyan cities
INSERT INTO properties (title, description, property_type_code, status_code, address_street, address_city, address_state, address_zip_code, address_country, bedrooms, bathrooms, area_sqft, monthly_rent, rent_amount, security_deposit, amenities, owner_id, is_published, created_by, created_at) VALUES
('Modern Apartment in Nakuru CBD', 'Beautiful 2-bedroom apartment in the heart of Nakuru with modern amenities and city views.', 'apartment', 'available', 'Kenyatta Avenue', 'Nakuru', 'Nakuru County', '20100', 'KE', 2, 2, 1000, 25000.00, 25000.00, 25000.00, '["wifi", "parking", "security", "water"]', 2, TRUE, 2, NOW()),
('Luxury Apartment in Westlands', 'Premium 3-bedroom apartment in Westlands, Nairobi with gym and swimming pool.', 'apartment', 'available', 'Westlands Road', 'Nairobi', 'Nairobi County', '00100', 'KE', 3, 2, 1400, 85000.00, 85000.00, 85000.00, '["wifi", "parking", "gym", "pool", "security"]', 2, TRUE, 2, NOW()),
('Beachfront Villa in Mombasa', 'Stunning 4-bedroom villa near Diani Beach with ocean views and private garden.', 'villa', 'available', 'Diani Beach Road', 'Mombasa', 'Mombasa County', '80100', 'KE', 4, 3, 2200, 120000.00, 120000.00, 120000.00, '["wifi", "parking", "garden", "beach_access", "security"]', 2, TRUE, 2, NOW()),
('Family House in Milimani', 'Spacious 3-bedroom house in quiet Milimani estate with garden and parking.', 'house', 'available', 'Milimani Estate', 'Nakuru', 'Nakuru County', '20100', 'KE', 3, 2, 1500, 35000.00, 35000.00, 35000.00, '["parking", "garden", "security", "water"]', 2, TRUE, 2, NOW()),
('Student Apartment in Eldoret', 'Affordable 1-bedroom apartment near Moi University, perfect for students.', 'apartment', 'available', 'University Road', 'Eldoret', 'Uasin Gishu County', '30100', 'KE', 1, 1, 600, 15000.00, 15000.00, 15000.00, '["wifi", "security", "water"]', 2, TRUE, 2, NOW());

-- Create sample driver
INSERT INTO drivers (user_id, license_number, license_expiry, vehicle_registration, is_verified, is_available, current_latitude, current_longitude, last_location_update) VALUES
(6, 'DL123456789', '2025-12-31', 'KCA123D', TRUE, TRUE, -0.3031, 36.0800, NOW());

-- Create sample vehicle
INSERT INTO transportation_vehicles (driver_id, vehicle_type, make, model, year, color, license_plate, capacity_weight, capacity_volume, is_active) VALUES
(1, 'pickup_truck', 'Toyota', 'Hilux', 2020, 'White', 'KCA123D', 1000.00, 5.00, TRUE);

-- Create sample leases
INSERT INTO leases (property_id, tenant_id, landlord_id, primary_tenant_id, lease_number, start_date, end_date, monthly_rent, security_deposit, status_code, created_by, created_at) VALUES
(1, 4, 2, 4, 'LEASE-001', '2024-01-01', '2024-12-31', 25000.00, 25000.00, 'active', 2, NOW()),
(2, 5, 2, 5, 'LEASE-002', '2024-02-01', '2025-01-31', 85000.00, 85000.00, 'active', 2, NOW()),
(4, 4, 2, 4, 'LEASE-003', '2024-03-01', '2025-02-28', 35000.00, 35000.00, 'active', 2, NOW());

-- Create sample payments
INSERT INTO payments (lease_id, tenant_id, landlord_id, property_id, amount, amount_due, amount_paid, period_month, period_year, due_date, paid_date, status_code, payment_method, payment_type, description, created_by, created_at) VALUES
(1, 4, 2, 1, 25000.00, 25000.00, 25000.00, 1, 2024, '2024-01-01', '2024-01-01', 'paid', 'bank_transfer', 'rent', 'January 2024 Rent', 2, NOW()),
(1, 4, 2, 1, 25000.00, 25000.00, 25000.00, 2, 2024, '2024-02-01', '2024-02-01', 'paid', 'bank_transfer', 'rent', 'February 2024 Rent', 2, NOW()),
(2, 5, 2, 2, 85000.00, 85000.00, 85000.00, 2, 2024, '2024-02-01', '2024-02-01', 'paid', 'bank_transfer', 'rent', 'February 2024 Rent', 2, NOW());

-- Create sample maintenance requests
INSERT INTO maintenance_requests (property_id, requested_by, tenant_id, landlord_id, title, description, category, priority_code, status_code, created_by, created_at) VALUES
(1, 4, 4, 2, 'Leaking Kitchen Faucet', 'The kitchen faucet has been leaking constantly for the past 2 days.', 'plumbing', 'normal', 'submitted', 4, NOW()),
(2, 5, 5, 2, 'AC Not Cooling', 'The air conditioning unit is running but not cooling the apartment.', 'hvac', 'high', 'in_progress', 5, NOW());

-- Create sample chat room
INSERT INTO chat_rooms (participant_1_id, participant_2_id, room_type, is_active, last_message_at) VALUES
(4, 2, 'direct', TRUE, NOW());

-- Create sample chat message
INSERT INTO chat_messages (room_id, sender_id, message_text, message_type, is_read) VALUES
(1, 4, 'Hello, I am interested in the apartment listing.', 'text', FALSE);

-- Create sample notifications
INSERT INTO notifications (user_id, title, message, type, is_read, created_at) VALUES
(4, 'Welcome to StaySpot!', 'Thank you for joining StaySpot. Your account has been successfully created.', 'system_alert', FALSE, NOW()),
(2, 'Payment Received', 'Your rent payment of KES 25,000.00 for January 2024 has been received.', 'payment_received', FALSE, NOW()),
(4, 'Maintenance Request Submitted', 'Your maintenance request "Leaking Kitchen Faucet" has been received and will be reviewed shortly.', 'maintenance_update', FALSE, NOW());

-- Create sample vendors
INSERT INTO vendors (company_name, contact_person, email, phone, service_category, description, address_city, address_state, address_country, is_approved, is_verified, hourly_rate, created_by, created_at) VALUES
('QuickFix Plumbing Kenya', 'Mike Johnson', 'mike@quickfixke.com', '+254712345001', 'plumbing', '24/7 emergency plumbing services with 15 years of experience', 'Nakuru', 'Nakuru County', 'KE', TRUE, TRUE, 2500.00, 1, NOW()),
('Pro Electrical Solutions', 'Sarah Wilson', 'sarah@proelectrical.co.ke', '+254723456002', 'electrical', 'Licensed and insured electrical contractors', 'Nairobi', 'Nairobi County', 'KE', TRUE, TRUE, 3000.00, 1, NOW());

-- Create sample emergency contacts
INSERT INTO emergency_contacts (property_id, contact_type, contact_name, phone_number, is_24_7) VALUES
(1, 'police', 'Nakuru Police Station', '+254202345678', TRUE),
(1, 'hospital', 'Nakuru Level 5 Hospital', '+254202345679', TRUE),
(2, 'police', 'Nairobi Central Police', '+254202123456', TRUE),
(2, 'hospital', 'Kenyatta National Hospital', '+254202726300', TRUE);

-- Create sample analytics data
INSERT INTO property_analytics (property_id, metric_date, views_count, inquiries_count, occupancy_rate, revenue) VALUES
(1, CURDATE() - INTERVAL 1 DAY, 45, 8, 100.00, 25000.00),
(2, CURDATE() - INTERVAL 1 DAY, 62, 12, 100.00, 85000.00);

-- ==================== CREATE VIEWS ====================

CREATE VIEW property_overview AS
SELECT 
    p.id,
    p.uuid,
    p.title,
    p.property_type_code,
    pt.type_name as property_type,
    p.status_code,
    ps.status_name as property_status,
    p.address_street,
    p.address_city,
    p.address_state,
    p.monthly_rent,
    p.bedrooms,
    p.bathrooms,
    p.area_sqft,
    p.available_from,
    CONCAT(u_owner.first_name, ' ', u_owner.last_name) as owner_name,
    CONCAT(u_manager.first_name, ' ', u_manager.last_name) as manager_name,
    (SELECT COUNT(*) FROM leases l WHERE l.property_id = p.id AND l.status_code = 'active' AND l.deleted_at IS NULL) as active_leases,
    (SELECT COUNT(*) FROM maintenance_requests mr WHERE mr.property_id = p.id AND mr.status_code != 'completed') as open_maintenance_requests
FROM properties p
LEFT JOIN property_types pt ON p.property_type_code = pt.type_code
LEFT JOIN property_statuses ps ON p.status_code = ps.status_code
LEFT JOIN users u_owner ON p.owner_id = u_owner.id
LEFT JOIN users u_manager ON p.property_manager_id = u_manager.id
WHERE p.deleted_at IS NULL;

CREATE VIEW driver_overview AS
SELECT 
    d.id,
    d.uuid,
    CONCAT(u.first_name, ' ', u.last_name) as driver_name,
    u.email,
    u.phone,
    d.license_number,
    d.is_verified,
    d.is_online,
    d.is_available,
    d.total_trips,
    d.completed_trips,
    d.average_rating,
    d.total_earnings,
    d.current_latitude,
    d.current_longitude,
    d.last_location_update,
    v.vehicle_type,
    v.make,
    v.model,
    v.license_plate
FROM drivers d
JOIN users u ON d.user_id = u.id
LEFT JOIN transportation_vehicles v ON d.id = v.driver_id AND v.is_active = TRUE;

CREATE VIEW active_leases AS
SELECT 
    l.*,
    p.title as property_title,
    CONCAT(p.address_street, ', ', p.address_city) as property_address,
    CONCAT(t.first_name, ' ', t.last_name) as tenant_name,
    t.email as tenant_email,
    CONCAT(ll.first_name, ' ', ll.last_name) as landlord_name
FROM leases l
JOIN properties p ON l.property_id = p.id
JOIN users t ON l.tenant_id = t.id
JOIN users ll ON l.landlord_id = ll.id
WHERE l.status_code = 'active' AND l.deleted_at IS NULL;

CREATE VIEW financial_summary AS
SELECT 
    p.id as property_id,
    p.title as property_title,
    p.monthly_rent,
    COALESCE(SUM(CASE WHEN pay.status_code = 'paid' THEN pay.amount_paid ELSE 0 END), 0) as total_rent_collected,
    COALESCE(SUM(CASE WHEN pay.status_code = 'overdue' THEN pay.amount_due ELSE 0 END), 0) as total_rent_overdue,
    COALESCE(SUM(e.amount), 0) as total_expenses,
    (COALESCE(SUM(CASE WHEN pay.status_code = 'paid' THEN pay.amount_paid ELSE 0 END), 0) - COALESCE(SUM(e.amount), 0)) as net_income
FROM properties p
LEFT JOIN leases l ON p.id = l.property_id AND l.status_code = 'active' AND l.deleted_at IS NULL
LEFT JOIN payments pay ON l.id = pay.lease_id
LEFT JOIN expenses e ON p.id = e.property_id AND YEAR(e.expense_date) = YEAR(CURRENT_DATE) AND MONTH(e.expense_date) = MONTH(CURRENT_DATE)
WHERE p.deleted_at IS NULL
GROUP BY p.id, p.title, p.monthly_rent;

CREATE VIEW maintenance_dashboard AS
SELECT 
    mr.id,
    mr.uuid,
    mr.title,
    mr.priority_code,
    mp.priority_name,
    mr.status_code,
    ms.status_name as maintenance_status,
    p.title as property_title,
    u_requester.first_name as requester_first_name,
    u_requester.last_name as requester_last_name,
    u_assigned.first_name as assigned_first_name,
    u_assigned.last_name as assigned_last_name,
    mr.requested_date,
    mr.scheduled_date,
    mr.completed_date,
    mr.estimated_cost,
    mr.actual_cost,
    DATEDIFF(CURRENT_DATE, mr.requested_date) as days_open
FROM maintenance_requests mr
LEFT JOIN maintenance_priorities mp ON mr.priority_code = mp.priority_code
LEFT JOIN maintenance_statuses ms ON mr.status_code = ms.status_code
LEFT JOIN properties p ON mr.property_id = p.id
LEFT JOIN users u_requester ON mr.requested_by = u_requester.id
LEFT JOIN users u_assigned ON mr.assigned_to = u_assigned.id;

-- ==================== CREATE INDEXES FOR PERFORMANCE ====================

CREATE INDEX idx_users_email_active ON users(email, is_active);
CREATE INDEX idx_properties_owner_available ON properties(owner_id, is_available);
CREATE INDEX idx_leases_dates ON leases(start_date, end_date);
CREATE INDEX idx_payments_dates ON payments(due_date, paid_date);
CREATE INDEX idx_maintenance_priority_status ON maintenance_requests(priority_code, status_code);

-- ==================== FINAL SETUP ====================

-- Display completion message
SELECT 'StaySpot Unified Database Schema Created Successfully!' as message;

-- Show table counts
SELECT 
    COUNT(*) as total_tables_created 
FROM information_schema.tables 
WHERE table_schema = 'stayspotco_stayspot' 
AND table_type = 'BASE TABLE';

-- Show sample data counts
SELECT 
    'Users' as table_name, COUNT(*) as count FROM users
UNION ALL
    SELECT 'Properties', COUNT(*) FROM properties
UNION ALL
    SELECT 'Leases', COUNT(*) FROM leases
UNION ALL
    SELECT 'Payments', COUNT(*) FROM payments
UNION ALL
    SELECT 'Maintenance Requests', COUNT(*) FROM maintenance_requests
UNION ALL
    SELECT 'Drivers', COUNT(*) FROM drivers
UNION ALL
    SELECT 'Chat Rooms', COUNT(*) FROM chat_rooms
UNION ALL
    SELECT 'Chat Messages', COUNT(*) FROM chat_messages
UNION ALL
    SELECT 'Transportation Vehicles', COUNT(*) FROM transportation_vehicles
UNION ALL
    SELECT 'Vendors', COUNT(*) FROM vendors
UNION ALL
    SELECT 'Notifications', COUNT(*) FROM notifications
UNION ALL
    SELECT 'Emergency Contacts', COUNT(*) FROM emergency_contacts
UNION ALL
    SELECT 'Property Analytics', COUNT(*) FROM property_analytics;

-- Show users by role
SELECT role_code, COUNT(*) as user_count 
FROM users 
GROUP BY role_code 
ORDER BY user_count DESC;

-- Show properties by status
SELECT status_code, COUNT(*) as property_count 
FROM properties 
GROUP BY status_code 
ORDER BY property_count DESC;

-- Show maintenance requests by priority
SELECT priority_code, COUNT(*) as request_count 
FROM maintenance_requests 
GROUP BY priority_code 
ORDER BY request_count DESC;