-- StaySpot Database Migration Script
-- From v2.0.0 to v3.0.0
-- Adds partnerships, chat, and transportation features

USE stayspot;

-- ==================== ADD NEW USER ROLE ====================

INSERT IGNORE INTO user_roles (role_code, role_name, description) VALUES
('driver', 'Transportation Driver', 'Moving and transportation services');

-- ==================== CREATE PARTNERSHIP TABLES ====================

CREATE TABLE IF NOT EXISTS partnership_applications (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Company Information
    company_name VARCHAR(255) NOT NULL,
    company_type VARCHAR(100) NOT NULL,
    industry VARCHAR(100) NOT NULL,
    website VARCHAR(255),
    
    -- Contact Information
    contact_name VARCHAR(100) NOT NULL,
    contact_email VARCHAR(255) NOT NULL,
    contact_phone VARCHAR(20),
    contact_position VARCHAR(100),
    
    -- Company Details
    company_size VARCHAR(50),
    annual_revenue VARCHAR(50),
    years_in_business INT,
    
    -- Address
    address_street VARCHAR(255),
    address_city VARCHAR(100),
    address_state VARCHAR(100),
    address_zip_code VARCHAR(20),
    address_country VARCHAR(100) DEFAULT 'KE',
    
    -- Partnership Details
    partnership_type VARCHAR(100) NOT NULL,
    services_offered TEXT,
    target_market TEXT,
    partnership_goals TEXT,
    
    -- Additional Information
    previous_partnerships TEXT,
    references TEXT,
    additional_info TEXT,
    
    -- Status
    status ENUM('pending', 'under_review', 'approved', 'rejected', 'on_hold') DEFAULT 'pending',
    reviewed_by BIGINT NULL,
    reviewed_at DATETIME NULL,
    review_notes TEXT,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_company_name (company_name),
    INDEX idx_contact_email (contact_email),
    INDEX idx_partnership_type (partnership_type),
    INDEX idx_status (status),
    INDEX idx_created_at (created_at),
    
    -- Foreign Keys
    FOREIGN KEY (reviewed_by) REFERENCES users(id)
);

-- ==================== CREATE CHAT TABLES ====================

CREATE TABLE IF NOT EXISTS chat_rooms (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Room Details
    room_name VARCHAR(255),
    room_type ENUM('direct', 'group', 'support') DEFAULT 'direct',
    
    -- Participants
    participant_1_id BIGINT NOT NULL,
    participant_2_id BIGINT NULL,
    participants JSON, -- For group chats
    
    -- Room Settings
    is_active BOOLEAN DEFAULT TRUE,
    last_message_at DATETIME,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_participant_1 (participant_1_id),
    INDEX idx_participant_2 (participant_2_id),
    INDEX idx_room_type (room_type),
    INDEX idx_active (is_active),
    INDEX idx_last_message (last_message_at),
    
    -- Foreign Keys
    FOREIGN KEY (participant_1_id) REFERENCES users(id),
    FOREIGN KEY (participant_2_id) REFERENCES users(id)
);

CREATE TABLE IF NOT EXISTS chat_messages (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Message Details
    room_id BIGINT NOT NULL,
    sender_id BIGINT NOT NULL,
    message_text TEXT NOT NULL,
    message_type ENUM('text', 'image', 'file', 'system') DEFAULT 'text',
    
    -- Attachments
    attachments JSON,
    
    -- Status
    is_read BOOLEAN DEFAULT FALSE,
    read_at DATETIME,
    is_edited BOOLEAN DEFAULT FALSE,
    edited_at DATETIME,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_room (room_id),
    INDEX idx_sender (sender_id),
    INDEX idx_created (created_at),
    INDEX idx_read (is_read),
    
    -- Foreign Keys
    FOREIGN KEY (room_id) REFERENCES chat_rooms(id) ON DELETE CASCADE,
    FOREIGN KEY (sender_id) REFERENCES users(id)
);

-- ==================== CREATE TRANSPORTATION TABLES ====================

CREATE TABLE IF NOT EXISTS drivers (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Driver Information (references users table)
    user_id BIGINT NOT NULL UNIQUE,
    
    -- Driver Specific Details
    license_number VARCHAR(100) NOT NULL,
    license_expiry DATE NOT NULL,
    vehicle_registration VARCHAR(100),
    
    -- Status
    is_verified BOOLEAN DEFAULT FALSE,
    is_online BOOLEAN DEFAULT FALSE,
    is_available BOOLEAN DEFAULT TRUE,
    
    -- Performance Metrics
    total_trips INT DEFAULT 0,
    completed_trips INT DEFAULT 0,
    cancelled_trips INT DEFAULT 0,
    average_rating DECIMAL(3,2) DEFAULT 0,
    total_earnings DECIMAL(10,2) DEFAULT 0,
    
    -- Current Location
    current_latitude DECIMAL(10,8),
    current_longitude DECIMAL(11,8),
    last_location_update DATETIME,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_user (user_id),
    INDEX idx_verified (is_verified),
    INDEX idx_online (is_online),
    INDEX idx_available (is_available),
    INDEX idx_location (current_latitude, current_longitude),
    INDEX idx_rating (average_rating),
    
    -- Foreign Keys
    FOREIGN KEY (user_id) REFERENCES users(id)
);

CREATE TABLE IF NOT EXISTS transportation_vehicles (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Vehicle Details
    driver_id BIGINT NOT NULL,
    vehicle_type ENUM('pickup_truck', 'van', 'box_truck', 'flatbed') NOT NULL,
    make VARCHAR(100) NOT NULL,
    model VARCHAR(100) NOT NULL,
    year YEAR NOT NULL,
    color VARCHAR(50),
    
    -- Registration
    license_plate VARCHAR(20) NOT NULL,
    registration_number VARCHAR(100),
    
    -- Specifications
    capacity_weight DECIMAL(8,2), -- in kg
    capacity_volume DECIMAL(8,2), -- in cubic meters
    
    -- Status
    is_active BOOLEAN DEFAULT TRUE,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_driver (driver_id),
    INDEX idx_vehicle_type (vehicle_type),
    INDEX idx_license_plate (license_plate),
    INDEX idx_active (is_active),
    
    -- Foreign Keys
    FOREIGN KEY (driver_id) REFERENCES drivers(id)
);

CREATE TABLE IF NOT EXISTS driver_bookings (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    uuid CHAR(36) NOT NULL UNIQUE DEFAULT (UUID()),
    
    -- Booking Details
    customer_id BIGINT NOT NULL,
    driver_id BIGINT NULL,
    vehicle_id BIGINT NULL,
    
    -- Service Details
    service_type ENUM('moving', 'delivery', 'transport') DEFAULT 'moving',
    vehicle_type ENUM('pickup_truck', 'van', 'box_truck', 'flatbed') NOT NULL,
    
    -- Locations
    pickup_address TEXT NOT NULL,
    pickup_latitude DECIMAL(10,8),
    pickup_longitude DECIMAL(11,8),
    dropoff_address TEXT NOT NULL,
    dropoff_latitude DECIMAL(10,8),
    dropoff_longitude DECIMAL(11,8),
    
    -- Timing
    scheduled_date DATE NOT NULL,
    scheduled_time TIME NOT NULL,
    estimated_duration INT, -- in minutes
    
    -- Pricing
    estimated_cost DECIMAL(10,2) NOT NULL,
    final_cost DECIMAL(10,2),
    distance_km DECIMAL(8,2),
    
    -- Status
    status ENUM('pending', 'confirmed', 'in_progress', 'completed', 'cancelled') DEFAULT 'pending',
    
    -- Additional Details
    special_instructions TEXT,
    items_description TEXT,
    
    -- Tracking
    started_at DATETIME,
    completed_at DATETIME,
    
    -- Rating & Feedback
    customer_rating INT,
    customer_feedback TEXT,
    driver_rating INT,
    driver_feedback TEXT,
    
    -- Audit Fields
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_customer (customer_id),
    INDEX idx_driver (driver_id),
    INDEX idx_vehicle (vehicle_id),
    INDEX idx_status (status),
    INDEX idx_scheduled (scheduled_date, scheduled_time),
    INDEX idx_service_type (service_type),
    INDEX idx_vehicle_type (vehicle_type),
    
    -- Foreign Keys
    FOREIGN KEY (customer_id) REFERENCES users(id),
    FOREIGN KEY (driver_id) REFERENCES drivers(id),
    FOREIGN KEY (vehicle_id) REFERENCES transportation_vehicles(id)
);

CREATE TABLE IF NOT EXISTS driver_locations (
    id BIGINT PRIMARY KEY AUTO_INCREMENT,
    
    -- Location Details
    driver_id BIGINT NOT NULL,
    booking_id BIGINT NULL,
    
    -- GPS Coordinates
    latitude DECIMAL(10,8) NOT NULL,
    longitude DECIMAL(11,8) NOT NULL,
    accuracy DECIMAL(8,2),
    
    -- Additional Info
    speed DECIMAL(5,2), -- km/h
    heading DECIMAL(5,2), -- degrees
    
    -- Timestamp
    recorded_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    
    -- Indexes
    INDEX idx_driver (driver_id),
    INDEX idx_booking (booking_id),
    INDEX idx_recorded (recorded_at),
    
    -- Foreign Keys
    FOREIGN KEY (driver_id) REFERENCES drivers(id),
    FOREIGN KEY (booking_id) REFERENCES driver_bookings(id)
);

-- ==================== UPDATE EXISTING TABLES ====================

-- Add new notification types
ALTER TABLE notifications 
MODIFY COLUMN type ENUM(
    'payment_reminder',
    'maintenance_update', 
    'lease_expiry',
    'new_message',
    'system_alert',
    'booking_confirmation',
    'payment_received',
    'maintenance_scheduled',
    'partnership_update',
    'driver_request'
) NOT NULL;

-- Add new entity types to notifications
ALTER TABLE notifications 
MODIFY COLUMN entity_type ENUM('property', 'lease', 'payment', 'maintenance', 'user', 'partnership', 'booking');

-- Add new payment methods (M-Pesa for Kenya)
ALTER TABLE payments 
MODIFY COLUMN payment_method ENUM('credit_card', 'debit_card', 'bank_transfer', 'cash', 'check', 'online', 'paypal', 'stripe', 'mpesa');

-- Update default country to Kenya
ALTER TABLE users 
MODIFY COLUMN address_country VARCHAR(100) DEFAULT 'KE';

ALTER TABLE properties 
MODIFY COLUMN address_country VARCHAR(100) DEFAULT 'KE';

-- Update audit logs entity types
ALTER TABLE audit_logs 
MODIFY COLUMN entity_type ENUM('user', 'property', 'lease', 'payment', 'maintenance', 'document', 'review', 'vendor', 'partnership', 'chat', 'booking') NOT NULL;

-- ==================== ADD NEW SYSTEM SETTINGS ====================

INSERT IGNORE INTO system_settings (setting_key, setting_value, setting_type, description, is_public) VALUES
('company_location', 'Nakuru, Kenya', 'string', 'Company headquarters location', TRUE),
('company_phone', '+254748041595', 'string', 'Company contact phone', TRUE),
('service_areas', 'Nairobi, Mombasa, Kisumu, Nakuru, Eldoret, Thika, Machakos, Meru, Nyeri, Kakamega, Kitale, Garissa, Malindi, Lamu, Naivasha, Nanyuki', 'string', 'Cities and towns where StaySpot operates', TRUE),
('default_currency', 'KES', 'string', 'Default currency for payments (Kenyan Shillings)', TRUE),
('chat_message_retention_days', '365', 'number', 'Days to retain chat messages', FALSE),
('driver_location_update_interval', '30', 'number', 'Seconds between location updates', FALSE),
('transportation_base_rate', '50', 'number', 'Base rate for transportation services (KES)', FALSE),
('transportation_per_km_rate', '25', 'number', 'Rate per kilometer (KES)', FALSE);

-- Update existing settings
UPDATE system_settings SET setting_value = '3.0.0' WHERE setting_key = 'platform_version';
UPDATE system_settings SET setting_value = 'KES' WHERE setting_key = 'default_currency';

-- ==================== CREATE NEW VIEWS ====================

CREATE OR REPLACE VIEW driver_overview AS
SELECT 
    d.id,
    d.uuid,
    CONCAT(u.first_name, ' ', u.last_name) as driver_name,
    u.email,
    u.phone,
    d.license_number,
    d.is_verified,
    d.is_online,
    d.is_available,
    d.total_trips,
    d.completed_trips,
    d.average_rating,
    d.total_earnings,
    d.current_latitude,
    d.current_longitude,
    d.last_location_update,
    v.vehicle_type,
    v.make,
    v.model,
    v.license_plate
FROM drivers d
JOIN users u ON d.user_id = u.id
LEFT JOIN transportation_vehicles v ON d.id = v.driver_id AND v.is_active = TRUE;

CREATE OR REPLACE VIEW partnership_summary AS
SELECT 
    pa.id,
    pa.uuid,
    pa.company_name,
    pa.company_type,
    pa.industry,
    pa.partnership_type,
    pa.contact_name,
    pa.contact_email,
    pa.status,
    pa.created_at,
    CONCAT(u.first_name, ' ', u.last_name) as reviewed_by_name,
    pa.reviewed_at
FROM partnership_applications pa
LEFT JOIN users u ON pa.reviewed_by = u.id;

-- ==================== INSERT SAMPLE DATA ====================

-- Add sample driver user (if not exists)
INSERT IGNORE INTO users (email, password_hash, first_name, last_name, phone, role_code, is_email_verified, is_active, company_name, address_country, created_at) VALUES
('driver@stayspot.co.ke', '$2a$12$LQv3c1yqBWVHxkd0L6kZrOeG5c4jU8VzLk8VZcFgKfM8nE9zL8bK', 'James', 'Kiprotich', '+254756789012', 'driver', TRUE, TRUE, 'StaySpot Transport', 'KE', NOW());

-- Get the driver user ID for sample data
SET @driver_user_id = (SELECT id FROM users WHERE email = 'driver@stayspot.co.ke' LIMIT 1);

-- Add sample driver (if user exists)
INSERT IGNORE INTO drivers (user_id, license_number, license_expiry, vehicle_registration, is_verified, is_available, current_latitude, current_longitude, last_location_update) 
SELECT @driver_user_id, 'DL123456789', '2025-12-31', 'KCA123D', TRUE, TRUE, -0.3031, 36.0800, NOW()
WHERE @driver_user_id IS NOT NULL;

-- Add sample vehicle (if driver exists)
INSERT IGNORE INTO transportation_vehicles (driver_id, vehicle_type, make, model, year, color, license_plate, capacity_weight, capacity_volume, is_active) 
SELECT 1, 'pickup_truck', 'Toyota', 'Hilux', 2020, 'White', 'KCA123D', 1000.00, 5.00, TRUE
WHERE EXISTS (SELECT 1 FROM drivers WHERE id = 1);

-- ==================== MIGRATION COMPLETE ====================

SELECT 'Database migration to v3.0.0 completed successfully!' as message;

-- Show service areas
SELECT setting_value as 'StaySpot Service Areas' FROM system_settings WHERE setting_key = 'service_areas';

-- Show new table counts
SELECT 
    'Partnership Applications' as table_name, COUNT(*) as count FROM partnership_applications
UNION ALL
    SELECT 'Chat Rooms', COUNT(*) FROM chat_rooms
UNION ALL
    SELECT 'Chat Messages', COUNT(*) FROM chat_messages
UNION ALL
    SELECT 'Drivers', COUNT(*) FROM drivers
UNION ALL
    SELECT 'Transportation Vehicles', COUNT(*) FROM transportation_vehicles
UNION ALL
    SELECT 'Driver Bookings', COUNT(*) FROM driver_bookings
UNION ALL
    SELECT 'Driver Locations', COUNT(*) FROM driver_locations;