/**
 * Additional Validation Functions
 */

const { isValidEmail, isValidPhone, isValidUrl } = require('./helpers');

// Validate phone number with specific format
const validatePhoneNumber = (phone, countryCode = 'US') => {
  if (!phone) return false;
  
  // Remove all non-digit characters
  const cleaned = phone.replace(/\D/g, '');
  
  // US/Canada phone validation (10 digits)
  if (countryCode === 'US' || countryCode === 'CA') {
    return cleaned.length === 10;
  }
  
  // International phone (minimum 8 digits, maximum 15)
  return cleaned.length >= 8 && cleaned.length <= 15;
};

// Validate ZIP code
const validateZipCode = (zipCode, countryCode = 'US') => {
  if (!zipCode) return false;
  
  if (countryCode === 'US') {
    const usZipRegex = /^\d{5}(-\d{4})?$/;
    return usZipRegex.test(zipCode);
  }
  
  // Basic validation for other countries (alphanumeric, 3-10 characters)
  const generalZipRegex = /^[a-zA-Z0-9\- ]{3,10}$/;
  return generalZipRegex.test(zipCode);
};

// Validate price/amount
const validatePrice = (price, allowZero = false) => {
  if (price === null || price === undefined) return false;
  
  const num = parseFloat(price);
  if (isNaN(num)) return false;
  
  if (allowZero) {
    return num >= 0;
  }
  
  return num > 0;
};

// Validate bedrooms count
const validateBedrooms = (bedrooms) => {
  const num = parseInt(bedrooms);
  return !isNaN(num) && num >= 0 && num <= 20;
};

// Validate bathrooms count
const validateBathrooms = (bathrooms) => {
  const num = parseFloat(bathrooms);
  return !isNaN(num) && num >= 0 && num <= 10;
};

// Validate property type
const validatePropertyType = (type) => {
  const validTypes = ['apartment', 'house', 'condo', 'townhouse', 'studio', 'villa'];
  return validTypes.includes(type);
};

// Validate lease status
const validateLeaseStatus = (status) => {
  const validStatuses = ['active', 'expired', 'terminated', 'pending'];
  return validStatuses.includes(status);
};

// Validate payment status
const validatePaymentStatus = (status) => {
  const validStatuses = ['pending', 'paid', 'overdue', 'failed', 'refunded'];
  return validStatuses.includes(status);
};

// Validate maintenance status
const validateMaintenanceStatus = (status) => {
  const validStatuses = ['pending', 'in_progress', 'completed', 'cancelled'];
  return validStatuses.includes(status);
};

// Validate maintenance priority
const validateMaintenancePriority = (priority) => {
  const validPriorities = ['low', 'medium', 'high', 'emergency'];
  return validPriorities.includes(priority);
};

// Validate maintenance category
const validateMaintenanceCategory = (category) => {
  const validCategories = [
    'plumbing', 'electrical', 'hvac', 'appliance', 
    'structural', 'cosmetic', 'pest_control', 'other'
  ];
  return validCategories.includes(category);
};

// Validate file type
const validateFileType = (mimetype, allowedTypes = null) => {
  const defaultAllowedTypes = [
    'image/jpeg',
    'image/jpg',
    'image/png',
    'image/webp',
    'application/pdf',
    'application/msword',
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
  ];
  
  const types = allowedTypes || defaultAllowedTypes;
  return types.includes(mimetype);
};

// Validate file size
const validateFileSize = (size, maxSize = 10 * 1024 * 1024) => {
  return size <= maxSize;
};

// Validate date range
const validateDateRange = (startDate, endDate) => {
  const start = new Date(startDate);
  const end = new Date(endDate);
  return start < end;
};

// Validate future date
const validateFutureDate = (date, includeToday = false) => {
  const today = new Date();
  today.setHours(0, 0, 0, 0);
  
  const inputDate = new Date(date);
  inputDate.setHours(0, 0, 0, 0);
  
  if (includeToday) {
    return inputDate >= today;
  }
  
  return inputDate > today;
};

// Validate past date
const validatePastDate = (date, includeToday = false) => {
  const today = new Date();
  today.setHours(0, 0, 0, 0);
  
  const inputDate = new Date(date);
  inputDate.setHours(0, 0, 0, 0);
  
  if (includeToday) {
    return inputDate <= today;
  }
  
  return inputDate < today;
};

// Validate credit card number (Luhn algorithm)
const validateCreditCardNumber = (cardNumber) => {
  // Remove all non-digit characters
  const cleaned = cardNumber.replace(/\D/g, '');
  
  // Check if it's a valid length
  if (cleaned.length < 13 || cleaned.length > 19) {
    return false;
  }
  
  // Luhn algorithm
  let sum = 0;
  let isEven = false;
  
  for (let i = cleaned.length - 1; i >= 0; i--) {
    let digit = parseInt(cleaned.charAt(i), 10);
    
    if (isEven) {
      digit *= 2;
      if (digit > 9) {
        digit -= 9;
      }
    }
    
    sum += digit;
    isEven = !isEven;
  }
  
  return sum % 10 === 0;
};

// Validate credit card expiry
const validateCreditCardExpiry = (month, year) => {
  const currentYear = new Date().getFullYear();
  const currentMonth = new Date().getMonth() + 1;
  
  const expMonth = parseInt(month, 10);
  const expYear = parseInt(year, 10);
  
  if (expYear < currentYear) return false;
  if (expYear === currentYear && expMonth < currentMonth) return false;
  if (expMonth < 1 || expMonth > 12) return false;
  
  return true;
};

// Validate CVV
const validateCVV = (cvv, cardType = null) => {
  if (!cvv) return false;
  
  const cleaned = cvv.replace(/\D/g, '');
  
  if (cardType === 'amex') {
    return cleaned.length === 4;
  }
  
  return cleaned.length === 3;
};

// Validate password strength
const validatePasswordStrength = (password) => {
  const minLength = 8;
  const hasUpperCase = /[A-Z]/.test(password);
  const hasLowerCase = /[a-z]/.test(password);
  const hasNumbers = /\d/.test(password);
  const hasSpecialChar = /[!@#$%^&*()_+\-=\[\]{};':"\\|,.<>\/?]/.test(password);
  
  return {
    isValid: password.length >= minLength && hasUpperCase && hasLowerCase && hasNumbers,
    strength: {
      length: password.length >= minLength,
      upperCase: hasUpperCase,
      lowerCase: hasLowerCase,
      numbers: hasNumbers,
      specialChar: hasSpecialChar
    },
    score: [
      password.length >= minLength,
      hasUpperCase,
      hasLowerCase,
      hasNumbers,
      hasSpecialChar
    ].filter(Boolean).length
  };
};

// Validate object ID (MongoDB style or numeric)
const validateObjectId = (id) => {
  if (!id) return false;
  
  // Check if it's a numeric ID
  if (!isNaN(id) && parseInt(id) > 0) return true;
  
  // Check if it's a MongoDB-style ID
  const mongoIdRegex = /^[0-9a-fA-F]{24}$/;
  return mongoIdRegex.test(id);
};

// Validate coordinates
const validateCoordinates = (latitude, longitude) => {
  const lat = parseFloat(latitude);
  const lng = parseFloat(longitude);
  
  if (isNaN(lat) || isNaN(lng)) return false;
  
  return lat >= -90 && lat <= 90 && lng >= -180 && lng <= 180;
};

// Validate US state code
const validateUSState = (stateCode) => {
  const validStates = [
    'AL', 'AK', 'AZ', 'AR', 'CA', 'CO', 'CT', 'DE', 'FL', 'GA',
    'HI', 'ID', 'IL', 'IN', 'IA', 'KS', 'KY', 'LA', 'ME', 'MD',
    'MA', 'MI', 'MN', 'MS', 'MO', 'MT', 'NE', 'NV', 'NH', 'NJ',
    'NM', 'NY', 'NC', 'ND', 'OH', 'OK', 'OR', 'PA', 'RI', 'SC',
    'SD', 'TN', 'TX', 'UT', 'VT', 'VA', 'WA', 'WV', 'WI', 'WY'
  ];
  
  return validStates.includes(stateCode.toUpperCase());
};

// Validate currency code
const validateCurrencyCode = (currencyCode) => {
  const validCurrencies = ['USD', 'EUR', 'GBP', 'CAD', 'AUD', 'JPY', 'CNY'];
  return validCurrencies.includes(currencyCode.toUpperCase());
};

// Validate timezone
const validateTimezone = (timezone) => {
  try {
    Intl.DateTimeFormat(undefined, { timeZone: timezone });
    return true;
  } catch (error) {
    return false;
  }
};

// Validate JSON string
const validateJSON = (str) => {
  try {
    JSON.parse(str);
    return true;
  } catch (error) {
    return false;
  }
};

// Validate array contents
const validateArray = (arr, validator) => {
  if (!Array.isArray(arr)) return false;
  
  for (const item of arr) {
    if (!validator(item)) return false;
  }
  
  return true;
};

// Validate object structure
const validateObjectStructure = (obj, structure) => {
  if (typeof obj !== 'object' || obj === null) return false;
  
  for (const key in structure) {
    if (structure.hasOwnProperty(key)) {
      const expectedType = structure[key];
      
      if (!obj.hasOwnProperty(key)) return false;
      
      if (expectedType === 'array') {
        if (!Array.isArray(obj[key])) return false;
      } else if (expectedType === 'object') {
        if (typeof obj[key] !== 'object' || obj[key] === null) return false;
      } else if (typeof obj[key] !== expectedType) {
        return false;
      }
    }
  }
  
  return true;
};

module.exports = {
  validatePhoneNumber,
  validateZipCode,
  validatePrice,
  validateBedrooms,
  validateBathrooms,
  validatePropertyType,
  validateLeaseStatus,
  validatePaymentStatus,
  validateMaintenanceStatus,
  validateMaintenancePriority,
  validateMaintenanceCategory,
  validateFileType,
  validateFileSize,
  validateDateRange,
  validateFutureDate,
  validatePastDate,
  validateCreditCardNumber,
  validateCreditCardExpiry,
  validateCVV,
  validatePasswordStrength,
  validateObjectId,
  validateCoordinates,
  validateUSState,
  validateCurrencyCode,
  validateTimezone,
  validateJSON,
  validateArray,
  validateObjectStructure
};