const axios = require('axios');

/**
 * Payment Processor Utility
 * Integrates with payment gateways like Stripe, PayPal, etc.
 */

class PaymentProcessor {
  constructor() {
    this.gateway = process.env.PAYMENT_GATEWAY || 'stripe';
    this.config = {
      stripe: {
        secretKey: process.env.STRIPE_SECRET_KEY,
        publishableKey: process.env.STRIPE_PUBLISHABLE_KEY
      },
      paypal: {
        clientId: process.env.PAYPAL_CLIENT_ID,
        clientSecret: process.env.PAYPAL_CLIENT_SECRET,
        environment: process.env.PAYPAL_ENVIRONMENT || 'sandbox'
      }
    };
  }

  // Process payment with selected gateway
  async processPayment(paymentData) {
    try {
      const { amount, currency, paymentMethod, description, metadata } = paymentData;

      switch (this.gateway) {
        case 'stripe':
          return await this.processStripePayment(paymentData);
        case 'paypal':
          return await this.processPayPalPayment(paymentData);
        case 'manual':
          return await this.processManualPayment(paymentData);
        default:
          throw new Error(`Unsupported payment gateway: ${this.gateway}`);
      }
    } catch (error) {
      console.error('Payment processing error:', error);
      throw error;
    }
  }

  // Process payment with Stripe
  async processStripePayment(paymentData) {
    const { amount, currency, paymentMethodId, description, metadata } = paymentData;

    try {
      const stripe = require('stripe')(this.config.stripe.secretKey);

      const paymentIntent = await stripe.paymentIntents.create({
        amount: Math.round(amount * 100), // Convert to cents
        currency: currency || 'usd',
        payment_method: paymentMethodId,
        confirm: true,
        description,
        metadata: {
          ...metadata,
          processed_by: 'stayspot'
        },
        return_url: `${process.env.FRONTEND_URL}/payment-success`
      });

      return {
        success: true,
        transactionId: paymentIntent.id,
        status: paymentIntent.status,
        amount: paymentIntent.amount / 100,
        currency: paymentIntent.currency,
        paymentMethod: 'stripe',
        rawResponse: paymentIntent
      };
    } catch (error) {
      return {
        success: false,
        error: error.message,
        declineCode: error.raw?.payment_intent?.last_payment_error?.decline_code,
        paymentMethod: 'stripe'
      };
    }
  }

  // Process payment with PayPal
  async processPayPalPayment(paymentData) {
    const { amount, currency, returnUrl, cancelUrl, description } = paymentData;

    try {
      // Get access token
      const auth = Buffer.from(
        `${this.config.paypal.clientId}:${this.config.paypal.clientSecret}`
      ).toString('base64');

      const baseUrl = this.config.paypal.environment === 'live' 
        ? 'https://api.paypal.com' 
        : 'https://api.sandbox.paypal.com';

      // Create order
      const orderResponse = await axios.post(
        `${baseUrl}/v2/checkout/orders`,
        {
          intent: 'CAPTURE',
          purchase_units: [
            {
              amount: {
                currency_code: currency || 'USD',
                value: amount.toString()
              },
              description
            }
          ],
          application_context: {
            return_url: returnUrl || `${process.env.FRONTEND_URL}/payment-success`,
            cancel_url: cancelUrl || `${process.env.FRONTEND_URL}/payment-cancel`
          }
        },
        {
          headers: {
            'Authorization': `Basic ${auth}`,
            'Content-Type': 'application/json'
          }
        }
      );

      return {
        success: true,
        orderId: orderResponse.data.id,
        status: orderResponse.data.status,
        approvalUrl: orderResponse.data.links.find(link => link.rel === 'approve').href,
        paymentMethod: 'paypal',
        rawResponse: orderResponse.data
      };
    } catch (error) {
      return {
        success: false,
        error: error.response?.data?.message || error.message,
        paymentMethod: 'paypal'
      };
    }
  }

  // Process manual payment (bank transfer, cash, check)
  async processManualPayment(paymentData) {
    const { amount, paymentMethod, referenceNumber, notes } = paymentData;

    // Generate a reference if not provided
    const reference = referenceNumber || `MANUAL-${Date.now()}-${Math.random().toString(36).substr(2, 9)}`;

    return {
      success: true,
      transactionId: reference,
      status: 'pending_verification',
      amount,
      paymentMethod: paymentMethod || 'manual',
      notes: notes || 'Manual payment awaiting verification',
      requiresVerification: true
    };
  }

  // Capture PayPal payment
  async capturePayPalPayment(orderId) {
    try {
      const auth = Buffer.from(
        `${this.config.paypal.clientId}:${this.config.paypal.clientSecret}`
      ).toString('base64');

      const baseUrl = this.config.paypal.environment === 'live' 
        ? 'https://api.paypal.com' 
        : 'https://api.sandbox.paypal.com';

      const captureResponse = await axios.post(
        `${baseUrl}/v2/checkout/orders/${orderId}/capture`,
        {},
        {
          headers: {
            'Authorization': `Basic ${auth}`,
            'Content-Type': 'application/json'
          }
        }
      );

      return {
        success: true,
        transactionId: captureResponse.data.id,
        status: captureResponse.data.status,
        amount: parseFloat(captureResponse.data.purchase_units[0].payments.captures[0].amount.value),
        currency: captureResponse.data.purchase_units[0].payments.captures[0].amount.currency_code,
        paymentMethod: 'paypal',
        rawResponse: captureResponse.data
      };
    } catch (error) {
      return {
        success: false,
        error: error.response?.data?.message || error.message,
        paymentMethod: 'paypal'
      };
    }
  }

  // Refund payment
  async refundPayment(transactionId, amount = null, reason = '') {
    try {
      switch (this.gateway) {
        case 'stripe':
          return await this.refundStripePayment(transactionId, amount, reason);
        case 'paypal':
          return await this.refundPayPalPayment(transactionId, amount, reason);
        default:
          return await this.refundManualPayment(transactionId, amount, reason);
      }
    } catch (error) {
      console.error('Refund error:', error);
      throw error;
    }
  }

  // Refund Stripe payment
  async refundStripePayment(transactionId, amount = null, reason = '') {
    const stripe = require('stripe')(this.config.stripe.secretKey);

    const refundData = {
      payment_intent: transactionId,
      reason: reason || 'requested_by_customer'
    };

    if (amount) {
      refundData.amount = Math.round(amount * 100);
    }

    const refund = await stripe.refunds.create(refundData);

    return {
      success: true,
      refundId: refund.id,
      status: refund.status,
      amount: refund.amount / 100,
      currency: refund.currency,
      reason: refund.reason
    };
  }

  // Refund PayPal payment
  async refundPayPalPayment(captureId, amount = null, reason = '') {
    const auth = Buffer.from(
      `${this.config.paypal.clientId}:${this.config.paypal.clientSecret}`
    ).toString('base64');

    const baseUrl = this.config.paypal.environment === 'live' 
      ? 'https://api.paypal.com' 
      : 'https://api.sandbox.paypal.com';

    const refundData = {
      amount: amount ? {
        value: amount.toString(),
        currency_code: 'USD'
      } : undefined,
      note_to_payer: reason
    };

    const refundResponse = await axios.post(
      `${baseUrl}/v2/payments/captures/${captureId}/refund`,
      refundData,
      {
        headers: {
          'Authorization': `Basic ${auth}`,
          'Content-Type': 'application/json'
        }
      }
    );

    return {
      success: true,
      refundId: refundResponse.data.id,
      status: refundResponse.data.status,
      amount: parseFloat(refundResponse.data.amount.value),
      currency: refundResponse.data.amount.currency_code
    };
  }

  // Process manual refund
  async refundManualPayment(transactionId, amount = null, reason = '') {
    return {
      success: true,
      refundId: `MANUAL-REFUND-${Date.now()}`,
      status: 'processed',
      amount,
      currency: 'USD',
      reason: reason || 'Manual refund processed',
      notes: 'This refund was processed manually by an administrator'
    };
  }

  // Verify payment status
  async verifyPayment(transactionId) {
    try {
      switch (this.gateway) {
        case 'stripe':
          const stripe = require('stripe')(this.config.stripe.secretKey);
          const paymentIntent = await stripe.paymentIntents.retrieve(transactionId);
          return {
            status: paymentIntent.status,
            amount: paymentIntent.amount / 100,
            currency: paymentIntent.currency,
            captured: paymentIntent.status === 'succeeded'
          };

        case 'paypal':
          const auth = Buffer.from(
            `${this.config.paypal.clientId}:${this.config.paypal.clientSecret}`
          ).toString('base64');

          const baseUrl = this.config.paypal.environment === 'live' 
            ? 'https://api.paypal.com' 
            : 'https://api.sandbox.paypal.com';

          const orderResponse = await axios.get(
            `${baseUrl}/v2/checkout/orders/${transactionId}`,
            {
              headers: {
                'Authorization': `Basic ${auth}`
              }
            }
          );

          return {
            status: orderResponse.data.status,
            amount: parseFloat(orderResponse.data.purchase_units[0].amount.value),
            currency: orderResponse.data.purchase_units[0].amount.currency_code,
            captured: orderResponse.data.status === 'COMPLETED'
          };

        default:
          return {
            status: 'unknown',
            requiresManualVerification: true
          };
      }
    } catch (error) {
      console.error('Payment verification error:', error);
      throw error;
    }
  }

  // Get supported payment methods
  getSupportedPaymentMethods() {
    return {
      stripe: ['credit_card', 'debit_card', 'bank_transfer'],
      paypal: ['paypal', 'credit_card'],
      manual: ['bank_transfer', 'cash', 'check']
    }[this.gateway] || ['manual'];
  }

  // Get payment gateway configuration
  getGatewayConfig() {
    return {
      gateway: this.gateway,
      config: this.config[this.gateway] || {},
      supportedMethods: this.getSupportedPaymentMethods()
    };
  }
}

module.exports = new PaymentProcessor();