const winston = require('winston');
const path = require('path');

/**
 * Logger Utility
 * Configures Winston logger for application logging
 */

// Define log format
const logFormat = winston.format.combine(
  winston.format.timestamp({
    format: 'YYYY-MM-DD HH:mm:ss'
  }),
  winston.format.errors({ stack: true }),
  winston.format.json()
);

// Define console format for development
const consoleFormat = winston.format.combine(
  winston.format.colorize(),
  winston.format.timestamp({
    format: 'YYYY-MM-DD HH:mm:ss'
  }),
  winston.format.printf(({ timestamp, level, message, stack, ...meta }) => {
    let log = `${timestamp} [${level}]: ${message}`;
    
    if (stack) {
      log += `\n${stack}`;
    }
    
    if (Object.keys(meta).length > 0) {
      log += `\n${JSON.stringify(meta, null, 2)}`;
    }
    
    return log;
  })
);

// Create the logger
const logger = winston.createLogger({
  level: process.env.LOG_LEVEL || 'info',
  format: logFormat,
  defaultMeta: {
    service: 'stayspot-backend'
  },
  transports: [
    // Write all logs with importance level of `error` or less to `error.log`
    new winston.transports.File({
      filename: path.join(__dirname, '../../logs/error.log'),
      level: 'error',
      maxsize: 5242880, // 5MB
      maxFiles: 5
    }),
    
    // Write all logs with importance level of `info` or less to `combined.log`
    new winston.transports.File({
      filename: path.join(__dirname, '../../logs/combined.log'),
      maxsize: 5242880, // 5MB
      maxFiles: 5
    })
  ]
});

// Add console transport for non-production environments
if (process.env.NODE_ENV !== 'production') {
  logger.add(new winston.transports.Console({
    format: consoleFormat
  }));
}

// Create a stream object for Morgan
logger.stream = {
  write: (message) => {
    logger.info(message.trim());
  }
};

// Custom logger methods
class Logger {
  // Log info message
  static info(message, meta = {}) {
    logger.info(message, meta);
  }

  // Log error message
  static error(message, error = null, meta = {}) {
    if (error instanceof Error) {
      logger.error(message, { ...meta, stack: error.stack });
    } else {
      logger.error(message, meta);
    }
  }

  // Log warning message
  static warn(message, meta = {}) {
    logger.warn(message, meta);
  }

  // Log debug message
  static debug(message, meta = {}) {
    logger.debug(message, meta);
  }

  // Log HTTP request
  static http(request, response, responseTime) {
    const logData = {
      method: request.method,
      url: request.url,
      ip: request.ip,
      userAgent: request.get('User-Agent'),
      userId: request.user?.id || 'anonymous',
      statusCode: response.statusCode,
      responseTime: `${responseTime}ms`
    };

    if (response.statusCode >= 400) {
      logger.warn('HTTP Request', logData);
    } else {
      logger.info('HTTP Request', logData);
    }
  }

  // Log database query
  static database(query, parameters = [], executionTime = null) {
    const logData = {
      query,
      parameters,
      executionTime: executionTime ? `${executionTime}ms` : null
    };

    logger.debug('Database Query', logData);
  }

  // Log authentication event
  static auth(event, userId, success = true, meta = {}) {
    const logData = {
      event,
      userId,
      success,
      ...meta
    };

    if (success) {
      logger.info('Authentication Event', logData);
    } else {
      logger.warn('Authentication Event', logData);
    }
  }

  // Log payment event
  static payment(event, paymentId, amount, status, meta = {}) {
    const logData = {
      event,
      paymentId,
      amount,
      status,
      ...meta
    };

    logger.info('Payment Event', logData);
  }

  // Log maintenance event
  static maintenance(event, maintenanceId, propertyId, status, meta = {}) {
    const logData = {
      event,
      maintenanceId,
      propertyId,
      status,
      ...meta
    };

    logger.info('Maintenance Event', logData);
  }

  // Log lease event
  static lease(event, leaseId, propertyId, tenantId, meta = {}) {
    const logData = {
      event,
      leaseId,
      propertyId,
      tenantId,
      ...meta
    };

    logger.info('Lease Event', logData);
  }

  // Log email event
  static email(event, to, subject, success = true, error = null) {
    const logData = {
      event,
      to,
      subject,
      success
    };

    if (error) {
      logData.error = error.message;
    }

    if (success) {
      logger.info('Email Event', logData);
    } else {
      logger.error('Email Event', logData);
    }
  }

  // Log file upload event
  static upload(event, fileName, fileSize, userId, success = true, error = null) {
    const logData = {
      event,
      fileName,
      fileSize,
      userId,
      success
    };

    if (error) {
      logData.error = error.message;
    }

    logger.info('File Upload Event', logData);
  }

  // Log security event
  static security(event, userId, ip, userAgent, meta = {}) {
    const logData = {
      event,
      userId,
      ip,
      userAgent,
      ...meta
    };

    logger.warn('Security Event', logData);
  }

  // Log performance metrics
  static performance(operation, duration, meta = {}) {
    const logData = {
      operation,
      duration: `${duration}ms`,
      ...meta
    };

    if (duration > 1000) {
      logger.warn('Performance Warning', logData);
    } else {
      logger.debug('Performance Metric', logData);
    }
  }

  // Create child logger with additional metadata
  static child(meta) {
    return logger.child(meta);
  }

  // Get logger instance (for direct Winston usage)
  static getLogger() {
    return logger;
  }
}

module.exports = Logger;