const { transporter, emailTemplates } = require('../config/email');
const { User } = require('../models');

/**
 * Email Service Utility
 */

class EmailService {
  // Send email with template
  async sendEmail(to, subject, html, attachments = []) {
    try {
      const mailOptions = {
        from: process.env.SMTP_FROM || 'noreply@stayspot.com',
        to,
        subject,
        html,
        attachments
      };

      const result = await transporter.sendMail(mailOptions);
      console.log(`Email sent to ${to}: ${result.messageId}`);
      return { success: true, messageId: result.messageId };
    } catch (error) {
      console.error('Email sending failed:', error);
      return { success: false, error: error.message };
    }
  }

  // Send welcome email to new user
  async sendWelcomeEmail(user) {
    const template = emailTemplates.WELCOME.template(user.firstName);
    return this.sendEmail(
      user.email,
      emailTemplates.WELCOME.subject,
      template
    );
  }

  // Send password reset email
  async sendPasswordResetEmail(user, resetToken) {
    const resetLink = `${process.env.FRONTEND_URL}/reset-password?token=${resetToken}`;
    const template = emailTemplates.PASSWORD_RESET.template(user.firstName, resetLink);
    
    return this.sendEmail(
      user.email,
      emailTemplates.PASSWORD_RESET.subject,
      template
    );
  }

  // Send lease creation notification
  async sendLeaseCreatedEmail(lease, tenant, landlord) {
    const template = emailTemplates.LEASE_CREATED.template(
      tenant.firstName,
      lease.property.address
    );
    
    return this.sendEmail(
      tenant.email,
      emailTemplates.LEASE_CREATED.subject,
      template
    );
  }

  // Send payment reminder
  async sendPaymentReminder(payment, tenant, daysUntilDue = 3) {
    const subject = `Payment Reminder: Rent Due in ${daysUntilDue} Days`;
    const html = `
      <h1>Payment Reminder</h1>
      <p>Hello ${tenant.firstName},</p>
      <p>This is a friendly reminder that your rent payment of $${payment.amount} is due in ${daysUntilDue} days.</p>
      <p><strong>Due Date:</strong> ${new Date(payment.dueDate).toLocaleDateString()}</p>
      <p>Please log in to your StaySpot account to make the payment.</p>
      <p>Thank you for your prompt attention to this matter.</p>
    `;

    return this.sendEmail(tenant.email, subject, html);
  }

  // Send overdue payment notification
  async sendOverduePaymentNotification(payment, tenant) {
    const subject = 'URGENT: Rent Payment Overdue';
    const html = `
      <h1>Overdue Payment Notification</h1>
      <p>Hello ${tenant.firstName},</p>
      <p>Your rent payment of $${payment.amount} was due on ${new Date(payment.dueDate).toLocaleDateString()} and is now overdue.</p>
      <p>Late fees may apply as per your lease agreement.</p>
      <p>Please make the payment immediately to avoid further action.</p>
      <p>If you have already made the payment, please disregard this notice.</p>
    `;

    return this.sendEmail(tenant.email, subject, html);
  }

  // Send maintenance update
  async sendMaintenanceUpdate(maintenance, user, updateMessage) {
    const subject = `Maintenance Update: ${maintenance.title}`;
    const html = `
      <h1>Maintenance Request Update</h1>
      <p>Hello ${user.firstName},</p>
      <p>Your maintenance request "${maintenance.title}" has been updated:</p>
      <p><strong>Status:</strong> ${maintenance.status}</p>
      <p><strong>Update:</strong> ${updateMessage}</p>
      <p>You can view the complete details in your StaySpot account.</p>
    `;

    return this.sendEmail(user.email, subject, html);
  }

  // Send lease expiry reminder
  async sendLeaseExpiryReminder(lease, tenant, daysUntilExpiry = 30) {
    const subject = `Lease Expiry Reminder: ${daysUntilExpiry} Days Notice`;
    const html = `
      <h1>Lease Expiry Reminder</h1>
      <p>Hello ${tenant.firstName},</p>
      <p>Your lease for ${lease.property.address} will expire in ${daysUntilExpiry} days.</p>
      <p><strong>Lease End Date:</strong> ${new Date(lease.endDate).toLocaleDateString()}</p>
      <p>Please contact your landlord to discuss renewal options or move-out procedures.</p>
    `;

    return this.sendEmail(tenant.email, subject, html);
  }

  // Send bulk emails to multiple users
  async sendBulkEmail(users, subject, htmlTemplate) {
    const results = [];
    
    for (const user of users) {
      const personalizedHtml = htmlTemplate
        .replace(/{{firstName}}/g, user.firstName)
        .replace(/{{lastName}}/g, user.lastName)
        .replace(/{{email}}/g, user.email);

      const result = await this.sendEmail(user.email, subject, personalizedHtml);
      results.push({ email: user.email, success: result.success });
      
      // Small delay to avoid overwhelming the email service
      await new Promise(resolve => setTimeout(resolve, 100));
    }

    return results;
  }

  // Verify email configuration
  async verifyEmailConfig() {
    try {
      await transporter.verify();
      return { success: true, message: 'Email configuration is valid' };
    } catch (error) {
      return { success: false, error: error.message };
    }
  }
}

module.exports = new EmailService();