const { sequelize } = require('../config/database');
const { QueryTypes } = require('sequelize');

/**
 * Database utility functions
 */

// Execute raw SQL query
const executeQuery = async (query, replacements = {}) => {
  try {
    const results = await sequelize.query(query, {
      replacements,
      type: QueryTypes.SELECT
    });
    return results;
  } catch (error) {
    console.error('Database query error:', error);
    throw error;
  }
};

// Execute raw SQL command (INSERT, UPDATE, DELETE)
const executeCommand = async (command, replacements = {}) => {
  try {
    const results = await sequelize.query(command, {
      replacements,
      type: QueryTypes.RAW
    });
    return results;
  } catch (error) {
    console.error('Database command error:', error);
    throw error;
  }
};

// Check if database connection is healthy
const checkDatabaseHealth = async () => {
  try {
    await sequelize.authenticate();
    
    // Check if we can execute a simple query
    await executeQuery('SELECT 1 as health_check');
    
    return {
      healthy: true,
      message: 'Database connection is healthy'
    };
  } catch (error) {
    return {
      healthy: false,
      message: `Database connection failed: ${error.message}`
    };
  }
};

// Get database statistics
const getDatabaseStats = async () => {
  try {
    const stats = await executeQuery(`
      SELECT 
        (SELECT COUNT(*) FROM users) as user_count,
        (SELECT COUNT(*) FROM properties) as property_count,
        (SELECT COUNT(*) FROM leases) as lease_count,
        (SELECT COUNT(*) FROM payments) as payment_count,
        (SELECT COUNT(*) FROM maintenance_requests) as maintenance_count,
        (SELECT COUNT(*) FROM notifications) as notification_count
    `);
    
    return stats[0] || {};
  } catch (error) {
    console.error('Error getting database stats:', error);
    return {};
  }
};

// Backup database (simplified version)
const backupDatabase = async () => {
  try {
    const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
    const backupFile = `backup-${timestamp}.sql`;
    
    // This is a simplified backup - in production, use proper backup tools
    console.log(`Database backup initiated: ${backupFile}`);
    
    return {
      success: true,
      message: 'Backup process initiated',
      backupFile,
      timestamp: new Date().toISOString()
    };
  } catch (error) {
    console.error('Backup error:', error);
    throw error;
  }
};

// Optimize database tables
const optimizeDatabase = async () => {
  try {
    const tables = ['users', 'properties', 'leases', 'payments', 'maintenance_requests'];
    
    for (const table of tables) {
      await executeCommand(`OPTIMIZE TABLE ${table}`);
    }
    
    return {
      success: true,
      message: 'Database optimization completed'
    };
  } catch (error) {
    console.error('Optimization error:', error);
    throw error;
  }
};

// Clean up old data
const cleanupOldData = async (days = 90) => {
  try {
    const cutoffDate = new Date();
    cutoffDate.setDate(cutoffDate.getDate() - days);

    // Clean up old audit logs
    await executeCommand(
      'DELETE FROM audit_logs WHERE created_at < ?',
      { replacements: [cutoffDate] }
    );

    // Clean up old notifications
    await executeCommand(
      'DELETE FROM notifications WHERE created_at < ? AND is_read = true',
      { replacements: [cutoffDate] }
    );

    return {
      success: true,
      message: `Cleaned up data older than ${days} days`
    };
  } catch (error) {
    console.error('Cleanup error:', error);
    throw error;
  }
};

module.exports = {
  executeQuery,
  executeCommand,
  checkDatabaseHealth,
  getDatabaseStats,
  backupDatabase,
  optimizeDatabase,
  cleanupOldData
};