const { Property, Lease, Payment, Maintenance, User, Review } = require('../models');
const { Op } = require('sequelize');
const moment = require('moment');

const reportService = {
  // Generate financial report
  generateFinancialReport: async (reportCriteria, userId, userRole) => {
    try {
      const { startDate, endDate, landlordId, propertyId, format = 'json' } = reportCriteria;

      // Check permissions
      const canGenerate = reportService.canGenerateFinancialReport(userRole);
      if (!canGenerate) {
        throw new Error('Not authorized to generate financial reports');
      }

      const where = {
        status: 'paid',
        paidDate: {
          [Op.between]: [new Date(startDate), new Date(endDate)]
        }
      };

      // Filter by landlord or property
      if (landlordId) where.landlordId = landlordId;
      if (propertyId) where.propertyId = propertyId;

      // For non-admin users, restrict to their own data
      if (!userRole.includes('admin') && userRole !== 'financial_officer') {
        where.landlordId = userId;
      }

      const payments = await Payment.findAll({
        where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['title', 'address', 'type']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['firstName', 'lastName']
          },
          {
            model: User,
            as: 'landlord',
            attributes: ['firstName', 'lastName', 'companyName']
          }
        ],
        order: [['paidDate', 'ASC']]
      });

      // Calculate summary statistics
      const summary = {
        totalRevenue: payments.reduce((sum, payment) => sum + parseFloat(payment.amount), 0),
        totalTransactions: payments.length,
        averagePayment: payments.length > 0 ? 
          payments.reduce((sum, payment) => sum + parseFloat(payment.amount), 0) / payments.length : 0,
        period: { startDate, endDate }
      };

      // Group by payment method
      const paymentMethods = {};
      payments.forEach(payment => {
        const method = payment.paymentMethod || 'unknown';
        if (!paymentMethods[method]) {
          paymentMethods[method] = { count: 0, total: 0 };
        }
        paymentMethods[method].count++;
        paymentMethods[method].total += parseFloat(payment.amount);
      });

      // Monthly breakdown
      const monthlyBreakdown = {};
      payments.forEach(payment => {
        const month = moment(payment.paidDate).format('YYYY-MM');
        if (!monthlyBreakdown[month]) {
          monthlyBreakdown[month] = { count: 0, total: 0 };
        }
        monthlyBreakdown[month].count++;
        monthlyBreakdown[month].total += parseFloat(payment.amount);
      });

      const reportData = {
        summary,
        payments,
        paymentMethods,
        monthlyBreakdown: Object.entries(monthlyBreakdown).map(([month, data]) => ({
          month,
          ...data
        }))
      };

      if (format === 'csv') {
        const csvData = payments.map(payment => ({
          Date: payment.paidDate.toISOString().split('T')[0],
          Property: payment.property.title,
          Tenant: `${payment.tenant.firstName} ${payment.tenant.lastName}`,
          Landlord: payment.landlord.companyName || `${payment.landlord.firstName} ${payment.landlord.lastName}`,
          Amount: payment.amount,
          Method: payment.paymentMethod || 'N/A',
          Type: payment.property.type
        }));

        return {
          success: true,
          data: {
            format: 'csv',
            data: csvData,
            filename: `financial-report-${moment().format('YYYY-MM-DD')}.csv`
          }
        };
      }

      return {
        success: true,
        data: {
          format: 'json',
          ...reportData
        }
      };
    } catch (error) {
      console.error('Report service - financial report error:', error);
      throw error;
    }
  },

  // Generate occupancy report
  generateOccupancyReport: async (reportCriteria, userId, userRole) => {
    try {
      const { asOfDate = new Date(), landlordId, propertyType } = reportCriteria;

      // Check permissions
      const canGenerate = reportService.canGenerateOccupancyReport(userRole);
      if (!canGenerate) {
        throw new Error('Not authorized to generate occupancy reports');
      }

      const where = {};
      if (landlordId) where.landlordId = landlordId;
      if (propertyType) where.type = propertyType;

      // For non-admin users, restrict to their own data
      if (!userRole.includes('admin')) {
        where.landlordId = userId;
      }

      const properties = await Property.findAll({
        where,
        include: [
          {
            model: Lease,
            as: 'leases',
            where: {
              status: 'active',
              startDate: { [Op.lte]: asOfDate },
              endDate: { [Op.gte]: asOfDate }
            },
            required: false,
            include: [{
              model: User,
              as: 'tenant',
              attributes: ['firstName', 'lastName', 'email']
            }]
          }
        ]
      });

      const reportData = properties.map(property => {
        const isOccupied = property.leases && property.leases.length > 0;
        const currentLease = isOccupied ? property.leases[0] : null;

        return {
          property: {
            id: property.id,
            title: property.title,
            address: property.address,
            type: property.type,
            price: property.price,
            bedrooms: property.bedrooms,
            bathrooms: property.bathrooms
          },
          status: isOccupied ? 'occupied' : 'vacant',
          currentLease: currentLease ? {
            tenant: `${currentLease.tenant.firstName} ${currentLease.tenant.lastName}`,
            startDate: currentLease.startDate,
            endDate: currentLease.endDate,
            monthlyRent: currentLease.monthlyRent
          } : null,
          daysVacant: isOccupied ? 0 : Math.ceil((new Date(asOfDate) - new Date(property.updatedAt)) / (1000 * 60 * 60 * 24))
        };
      });

      // Calculate occupancy statistics
      const totalProperties = properties.length;
      const occupiedProperties = reportData.filter(item => item.status === 'occupied').length;
      const vacancyRate = totalProperties > 0 ? ((totalProperties - occupiedProperties) / totalProperties) * 100 : 0;
      const averageRent = reportData
        .filter(item => item.status === 'occupied')
        .reduce((sum, item) => sum + parseFloat(item.currentLease.monthlyRent), 0) / occupiedProperties || 0;

      const summary = {
        totalProperties,
        occupiedProperties,
        vacantProperties: totalProperties - occupiedProperties,
        occupancyRate: ((occupiedProperties / totalProperties) * 100).toFixed(2),
        vacancyRate: vacancyRate.toFixed(2),
        averageRent: averageRent.toFixed(2),
        asOfDate
      };

      return {
        success: true,
        data: {
          summary,
          properties: reportData
        }
      };
    } catch (error) {
      console.error('Report service - occupancy report error:', error);
      throw error;
    }
  },

  // Generate maintenance report
  generateMaintenanceReport: async (reportCriteria, userId, userRole) => {
    try {
      const { startDate, endDate, propertyId, status, priority, category } = reportCriteria;

      // Check permissions
      const canGenerate = reportService.canGenerateMaintenanceReport(userRole);
      if (!canGenerate) {
        throw new Error('Not authorized to generate maintenance reports');
      }

      const where = {
        createdAt: {
          [Op.between]: [new Date(startDate), new Date(endDate)]
        }
      };

      if (propertyId) where.propertyId = propertyId;
      if (status) where.status = status;
      if (priority) where.priority = priority;
      if (category) where.category = category;

      // For non-admin users, restrict to their data
      if (!userRole.includes('admin') && !userRole.includes('maintenance')) {
        where.landlordId = userId;
      }

      const maintenanceRequests = await Maintenance.findAll({
        where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['title', 'address']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['firstName', 'lastName']
          },
          {
            model: User,
            as: 'assignedStaff',
            attributes: ['firstName', 'lastName']
          },
          {
            model: User,
            as: 'landlord',
            attributes: ['firstName', 'lastName']
          }
        ],
        order: [['createdAt', 'DESC']]
      });

      // Calculate statistics
      const totalRequests = maintenanceRequests.length;
      const completedRequests = maintenanceRequests.filter(req => req.status === 'completed').length;
      const pendingRequests = maintenanceRequests.filter(req => req.status === 'pending').length;
      const inProgressRequests = maintenanceRequests.filter(req => req.status === 'in_progress').length;

      const totalCost = maintenanceRequests
        .filter(req => req.cost)
        .reduce((sum, req) => sum + parseFloat(req.cost), 0);

      // Category breakdown
      const categoryBreakdown = {};
      maintenanceRequests.forEach(req => {
        const category = req.category;
        if (!categoryBreakdown[category]) {
          categoryBreakdown[category] = { count: 0, totalCost: 0 };
        }
        categoryBreakdown[category].count++;
        if (req.cost) {
          categoryBreakdown[category].totalCost += parseFloat(req.cost);
        }
      });

      // Priority breakdown
      const priorityBreakdown = {};
      maintenanceRequests.forEach(req => {
        const priority = req.priority;
        if (!priorityBreakdown[priority]) {
          priorityBreakdown[priority] = { count: 0 };
        }
        priorityBreakdown[priority].count++;
      });

      const summary = {
        totalRequests,
        completedRequests,
        pendingRequests,
        inProgressRequests,
        completionRate: totalRequests > 0 ? ((completedRequests / totalRequests) * 100).toFixed(2) : 0,
        totalCost: totalCost.toFixed(2),
        averageCost: completedRequests > 0 ? (totalCost / completedRequests).toFixed(2) : 0,
        period: { startDate, endDate }
      };

      return {
        success: true,
        data: {
          summary,
          maintenanceRequests,
          categoryBreakdown,
          priorityBreakdown
        }
      };
    } catch (error) {
      console.error('Report service - maintenance report error:', error);
      throw error;
    }
  },

  // Generate tenant report
  generateTenantReport: async (reportCriteria, userId, userRole) => {
    try {
      const { landlordId, status = 'active' } = reportCriteria;

      // Check permissions
      const canGenerate = reportService.canGenerateTenantReport(userRole);
      if (!canGenerate) {
        throw new Error('Not authorized to generate tenant reports');
      }

      const where = { status };
      if (landlordId) where.landlordId = landlordId;

      // For non-admin users, restrict to their own data
      if (!userRole.includes('admin')) {
        where.landlordId = userId;
      }

      const leases = await Lease.findAll({
        where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['title', 'address', 'type', 'bedrooms', 'bathrooms']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['firstName', 'lastName', 'email', 'phone', 'createdAt']
          },
          {
            model: Payment,
            as: 'payments',
            attributes: ['amount', 'status', 'dueDate', 'paidDate']
          }
        ],
        order: [['endDate', 'ASC']]
      });

      const tenantData = leases.map(lease => {
        const payments = lease.payments || [];
        const totalPaid = payments
          .filter(p => p.status === 'paid')
          .reduce((sum, p) => sum + parseFloat(p.amount), 0);

        const pendingPayments = payments.filter(p => p.status === 'pending');
        const overduePayments = pendingPayments.filter(p => new Date(p.dueDate) < new Date());

        return {
          tenant: {
            id: lease.tenant.id,
            name: `${lease.tenant.firstName} ${lease.tenant.lastName}`,
            email: lease.tenant.email,
            phone: lease.tenant.phone,
            memberSince: lease.tenant.createdAt
          },
          lease: {
            id: lease.id,
            startDate: lease.startDate,
            endDate: lease.endDate,
            monthlyRent: lease.monthlyRent,
            securityDeposit: lease.securityDeposit
          },
          property: lease.property,
          financials: {
            totalPaid,
            pendingPayments: pendingPayments.length,
            overduePayments: overduePayments.length,
            totalOverdue: overduePayments.reduce((sum, p) => sum + parseFloat(p.amount), 0)
          },
          daysRemaining: Math.ceil((new Date(lease.endDate) - new Date()) / (1000 * 60 * 60 * 24))
        };
      });

      // Calculate summary statistics
      const totalTenants = tenantData.length;
      const totalMonthlyRent = tenantData.reduce((sum, tenant) => sum + parseFloat(tenant.lease.monthlyRent), 0);
      const totalOverdue = tenantData.reduce((sum, tenant) => sum + tenant.financials.totalOverdue, 0);
      const tenantsWithOverdue = tenantData.filter(tenant => tenant.financials.overduePayments > 0).length;

      const summary = {
        totalTenants,
        totalMonthlyRent: totalMonthlyRent.toFixed(2),
        averageRent: totalTenants > 0 ? (totalMonthlyRent / totalTenants).toFixed(2) : 0,
        totalOverdue: totalOverdue.toFixed(2),
        tenantsWithOverdue,
        overdueRate: totalTenants > 0 ? ((tenantsWithOverdue / totalTenants) * 100).toFixed(2) : 0
      };

      return {
        success: true,
        data: {
          summary,
          tenants: tenantData
        }
      };
    } catch (error) {
      console.error('Report service - tenant report error:', error);
      throw error;
    }
  },

  // Generate performance report
  generatePerformanceReport: async (reportCriteria, userId, userRole) => {
    try {
      const { year = moment().year(), landlordId } = reportCriteria;

      // Check permissions
      const canGenerate = reportService.canGeneratePerformanceReport(userRole);
      if (!canGenerate) {
        throw new Error('Not authorized to generate performance reports');
      }

      const where = {};
      if (landlordId) where.landlordId = landlordId;

      // For non-admin users, restrict to their own data
      if (!userRole.includes('admin')) {
        where.landlordId = userId;
      }

      // Monthly revenue data
      const monthlyRevenue = [];
      for (let month = 1; month <= 12; month++) {
        const startDate = new Date(year, month - 1, 1);
        const endDate = new Date(year, month, 0);

        const revenue = await Payment.sum('amount', {
          where: {
            ...where,
            status: 'paid',
            paidDate: {
              [Op.between]: [startDate, endDate]
            }
          }
        });

        monthlyRevenue.push({
          month: startDate.toLocaleDateString('en-US', { month: 'short' }),
          revenue: revenue || 0
        });
      }

      // Property performance
      const propertyPerformance = await Payment.findAll({
        where: {
          ...where,
          status: 'paid',
          paidDate: {
            [Op.gte]: new Date(year, 0, 1),
            [Op.lt]: new Date(year + 1, 0, 1)
          }
        },
        attributes: [
          'propertyId',
          [Payment.sequelize.fn('SUM', Payment.sequelize.col('amount')), 'totalRevenue'],
          [Payment.sequelize.fn('COUNT', Payment.sequelize.col('id')), 'paymentCount']
        ],
        group: ['propertyId'],
        include: [{
          model: Property,
          as: 'property',
          attributes: ['title', 'address', 'type']
        }],
        order: [[Payment.sequelize.fn('SUM', Payment.sequelize.col('amount')), 'DESC']]
      });

      // Lease performance
      const leaseStats = await Lease.findAll({
        where: {
          ...where,
          createdAt: {
            [Op.gte]: new Date(year, 0, 1),
            [Op.lt]: new Date(year + 1, 0, 1)
          }
        },
        attributes: [
          [Lease.sequelize.fn('COUNT', Lease.sequelize.col('id')), 'totalLeases'],
          [Lease.sequelize.fn('SUM', Lease.sequelize.col('monthlyRent')), 'totalRentValue']
        ]
      });

      const totalLeases = leaseStats[0]?.dataValues.totalLeases || 0;
      const totalRentValue = leaseStats[0]?.dataValues.totalRentValue || 0;

      return {
        success: true,
        data: {
          period: { year },
          monthlyRevenue,
          propertyPerformance,
          summary: {
            totalLeases,
            totalRentValue,
            averageLeaseValue: totalLeases > 0 ? (totalRentValue / totalLeases).toFixed(2) : 0
          }
        }
      };
    } catch (error) {
      console.error('Report service - performance report error:', error);
      throw error;
    }
  },

  // Permission check methods
  canGenerateFinancialReport: (userRole) => {
    const allowedRoles = [
      'super_admin',
      'system_admin',
      'landlord',
      'financial_officer',
      'financial_analyst',
      'property_manager',
      'data_analyst'
    ];
    return allowedRoles.includes(userRole);
  },

  canGenerateOccupancyReport: (userRole) => {
    const allowedRoles = [
      'super_admin',
      'system_admin',
      'landlord',
      'property_manager',
      'regional_manager',
      'leasing_consultant',
      'data_analyst'
    ];
    return allowedRoles.includes(userRole);
  },

  canGenerateMaintenanceReport: (userRole) => {
    const allowedRoles = [
      'super_admin',
      'system_admin',
      'landlord',
      'property_manager',
      'maintenance_supervisor',
      'community_manager',
      'data_analyst'
    ];
    return allowedRoles.includes(userRole);
  },

  canGenerateTenantReport: (userRole) => {
    const allowedRoles = [
      'super_admin',
      'system_admin',
      'landlord',
      'property_manager',
      'leasing_consultant',
      'customer_support_agent'
    ];
    return allowedRoles.includes(userRole);
  },

  canGeneratePerformanceReport: (userRole) => {
    const allowedRoles = [
      'super_admin',
      'system_admin',
      'landlord',
      'property_manager',
      'regional_manager',
      'data_analyst'
    ];
    return allowedRoles.includes(userRole);
  }
};

module.exports = reportService;