const { Payment, Lease, Property, User, AuditLog } = require('../models');
const { Op } = require('sequelize');
const moment = require('moment');
const paymentProcessor = require('../utils/paymentProcessor');
const emailService = require('../utils/emailService');

const paymentService = {
  // Get all payments with filtering
  getPayments: async (filters = {}) => {
    try {
      const {
        page = 1,
        limit = 10,
        status,
        leaseId,
        propertyId,
        tenantId,
        landlordId,
        month,
        year,
        sortBy = 'dueDate',
        sortOrder = 'DESC'
      } = filters;

      const offset = (page - 1) * limit;
      const where = {};

      // Build filter conditions
      if (status) where.status = status;
      if (leaseId) where.leaseId = leaseId;
      if (propertyId) where.propertyId = propertyId;
      if (tenantId) where.tenantId = tenantId;
      if (landlordId) where.landlordId = landlordId;

      // Filter by month/year
      if (month && year) {
        const startDate = new Date(year, month - 1, 1);
        const endDate = new Date(year, month, 0);
        where.dueDate = {
          [Op.between]: [startDate, endDate]
        };
      }

      const payments = await Payment.findAndCountAll({
        where,
        include: [
          {
            model: Lease,
            as: 'lease',
            attributes: ['id', 'startDate', 'endDate']
          },
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email']
          },
          {
            model: User,
            as: 'landlord',
            attributes: ['id', 'firstName', 'lastName', 'email']
          }
        ],
        order: [[sortBy, sortOrder.toUpperCase()]],
        limit: parseInt(limit),
        offset: parseInt(offset)
      });

      return {
        success: true,
        data: {
          payments: payments.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: payments.count,
            pages: Math.ceil(payments.count / limit)
          }
        }
      };
    } catch (error) {
      console.error('Payment service - get payments error:', error);
      throw error;
    }
  },

  // Get payment by ID
  getPaymentById: async (paymentId, userId, userRole) => {
    try {
      const payment = await Payment.findByPk(paymentId, {
        include: [
          {
            model: Lease,
            as: 'lease',
            include: ['property']
          },
          {
            model: Property,
            as: 'property'
          },
          {
            model: User,
            as: 'tenant',
            attributes: { exclude: ['password'] }
          },
          {
            model: User,
            as: 'landlord',
            attributes: { exclude: ['password'] }
          }
        ]
      });

      if (!payment) {
        throw new Error('Payment not found');
      }

      // Check if user can view this payment
      const canView = await paymentService.canViewPayment(payment, userId, userRole);
      if (!canView) {
        throw new Error('Not authorized to view this payment');
      }

      return {
        success: true,
        data: { payment }
      };
    } catch (error) {
      console.error('Payment service - get payment error:', error);
      throw error;
    }
  },

  // Process payment
  processPayment: async (paymentId, paymentData, userId, userRole, ipAddress, userAgent) => {
    try {
      const payment = await Payment.findByPk(paymentId, {
        include: ['property', 'tenant', 'landlord']
      });

      if (!payment) {
        throw new Error('Payment not found');
      }

      if (payment.status !== 'pending') {
        throw new Error('Payment has already been processed');
      }

      // Check if user can process this payment
      const canProcess = await paymentService.canProcessPayment(payment, userId, userRole);
      if (!canProcess) {
        throw new Error('Not authorized to process this payment');
      }

      // Process payment through payment processor
      const paymentResult = await paymentProcessor.processPayment({
        amount: payment.amount,
        paymentMethod: paymentData.paymentMethod,
        transactionDetails: paymentData.transactionDetails,
        description: `Rent payment for ${payment.property.title}`
      });

      if (!paymentResult.success) {
        throw new Error(paymentResult.message);
      }

      const oldValues = { ...payment.toJSON() };

      // Update payment record
      await payment.update({
        status: 'paid',
        paidDate: new Date(),
        paymentMethod: paymentData.paymentMethod,
        transactionId: paymentResult.transactionId,
        receiptUrl: paymentResult.receiptUrl
      });

      // Log the payment processing
      await AuditLog.create({
        action: 'payment_processed',
        entityType: 'payment',
        entityId: payment.id,
        userId: userId,
        oldValues,
        newValues: payment.toJSON(),
        description: `Payment processed for ${payment.property.title} - Amount: $${payment.amount}`,
        ipAddress,
        userAgent
      });

      // Send payment confirmation email to tenant
      await emailService.sendPaymentConfirmation(payment.tenant, payment);

      // Send payment received email to landlord
      await emailService.sendPaymentReceivedEmail(payment.landlord, payment);

      const updatedPayment = await Payment.findByPk(paymentId, {
        include: ['lease', 'property', 'tenant', 'landlord']
      });

      return {
        success: true,
        data: { payment: updatedPayment }
      };
    } catch (error) {
      console.error('Payment service - process payment error:', error);
      throw error;
    }
  },

  // Record manual payment
  recordManualPayment: async (paymentId, paymentData, userId, userRole, ipAddress, userAgent) => {
    try {
      const payment = await Payment.findByPk(paymentId, {
        include: ['property', 'tenant', 'landlord']
      });

      if (!payment) {
        throw new Error('Payment not found');
      }

      if (payment.status !== 'pending') {
        throw new Error('Payment has already been processed');
      }

      // Check if user can record manual payment
      const canRecordManual = await paymentService.canRecordManualPayment(payment, userId, userRole);
      if (!canRecordManual) {
        throw new Error('Not authorized to record manual payment');
      }

      const oldValues = { ...payment.toJSON() };

      await payment.update({
        status: 'paid',
        paidDate: paymentData.paidDate ? new Date(paymentData.paidDate) : new Date(),
        paymentMethod: paymentData.paymentMethod || 'cash',
        description: paymentData.notes ? `${payment.description} - ${paymentData.notes}` : payment.description
      });

      // Log the manual payment recording
      await AuditLog.create({
        action: 'manual_payment_recorded',
        entityType: 'payment',
        entityId: payment.id,
        userId: userId,
        oldValues,
        newValues: payment.toJSON(),
        description: `Manual payment recorded for ${payment.property.title} - Amount: $${payment.amount}`,
        ipAddress,
        userAgent
      });

      const updatedPayment = await Payment.findByPk(paymentId, {
        include: ['lease', 'property', 'tenant', 'landlord']
      });

      return {
        success: true,
        data: { payment: updatedPayment }
      };
    } catch (error) {
      console.error('Payment service - record manual payment error:', error);
      throw error;
    }
  },

  // Get payment statistics
  getPaymentStats: async (userId, userRole, filters = {}) => {
    try {
      const { year = moment().year() } = filters;
      let stats = {};

      if (userRole === 'landlord' || userRole === 'bnb_host') {
        stats = await paymentService.getLandlordPaymentStats(userId, year);
      } else if (userRole === 'tenant') {
        stats = await paymentService.getTenantPaymentStats(userId);
      } else if (userRole.includes('financial')) {
        stats = await paymentService.getFinancialPaymentStats(year);
      } else if (userRole.includes('admin')) {
        stats = await paymentService.getAdminPaymentStats(year);
      }

      return {
        success: true,
        data: { stats }
      };
    } catch (error) {
      console.error('Payment service - get payment stats error:', error);
      throw error;
    }
  },

  // Landlord payment statistics
  getLandlordPaymentStats: async (landlordId, year) => {
    const totalRevenue = await Payment.sum('amount', {
      where: {
        landlordId,
        status: 'paid',
        paidDate: {
          [Op.gte]: new Date(year, 0, 1),
          [Op.lt]: new Date(year + 1, 0, 1)
        }
      }
    });

    const pendingPayments = await Payment.sum('amount', {
      where: {
        landlordId,
        status: 'pending',
        dueDate: { [Op.lte]: new Date() }
      }
    });

    const paymentStatusCounts = await Payment.findAll({
      where: { landlordId },
      attributes: [
        'status',
        [Payment.sequelize.fn('COUNT', Payment.sequelize.col('id')), 'count'],
        [Payment.sequelize.fn('SUM', Payment.sequelize.col('amount')), 'total']
      ],
      group: ['status']
    });

    // Monthly revenue
    const monthlyRevenue = [];
    for (let month = 1; month <= 12; month++) {
      const startDate = new Date(year, month - 1, 1);
      const endDate = new Date(year, month, 0);

      const revenue = await Payment.sum('amount', {
        where: {
          landlordId,
          status: 'paid',
          paidDate: {
            [Op.between]: [startDate, endDate]
          }
        }
      });

      monthlyRevenue.push({
        month: startDate.toLocaleDateString('en-US', { month: 'short' }),
        revenue: revenue || 0
      });
    }

    return {
      totalRevenue: totalRevenue || 0,
      pendingPayments: pendingPayments || 0,
      paymentStatusCounts,
      monthlyRevenue
    };
  },

  // Tenant payment statistics
  getTenantPaymentStats: async (tenantId) => {
    const totalPaid = await Payment.sum('amount', {
      where: {
        tenantId,
        status: 'paid'
      }
    });

    const upcomingPayment = await Payment.findOne({
      where: {
        tenantId,
        status: 'pending',
        dueDate: { [Op.gte]: new Date() }
      },
      order: [['dueDate', 'ASC']],
      include: ['property']
    });

    const overduePayments = await Payment.sum('amount', {
      where: {
        tenantId,
        status: 'pending',
        dueDate: { [Op.lt]: new Date() }
      }
    });

    const paymentHistory = await Payment.findAll({
      where: { tenantId },
      order: [['dueDate', 'DESC']],
      limit: 6,
      include: ['property']
    });

    return {
      totalPaid: totalPaid || 0,
      upcomingPayment,
      overduePayments: overduePayments || 0,
      paymentHistory
    };
  },

  // Financial role payment statistics
  getFinancialPaymentStats: async (year) => {
    const totalRevenue = await Payment.sum('amount', {
      where: {
        status: 'paid',
        paidDate: {
          [Op.gte]: new Date(year, 0, 1),
          [Op.lt]: new Date(year + 1, 0, 1)
        }
      }
    });

    const pendingRevenue = await Payment.sum('amount', {
      where: { status: 'pending' }
    });

    const processedThisMonth = await Payment.count({
      where: {
        status: 'paid',
        paidDate: {
          [Op.gte]: new Date(new Date().getFullYear(), new Date().getMonth(), 1)
        }
      }
    });

    const paymentMethods = await Payment.findAll({
      where: { status: 'paid' },
      attributes: [
        'paymentMethod',
        [Payment.sequelize.fn('COUNT', Payment.sequelize.col('id')), 'count'],
        [Payment.sequelize.fn('SUM', Payment.sequelize.col('amount')), 'total']
      ],
      group: ['paymentMethod']
    });

    return {
      totalRevenue: totalRevenue || 0,
      pendingRevenue: pendingRevenue || 0,
      processedThisMonth,
      paymentMethods
    };
  },

  // Admin payment statistics
  getAdminPaymentStats: async (year) => {
    const totalRevenue = await Payment.sum('amount', {
      where: {
        status: 'paid',
        paidDate: {
          [Op.gte]: new Date(year, 0, 1),
          [Op.lt]: new Date(year + 1, 0, 1)
        }
      }
    });

    const totalPayments = await Payment.count({
      where: {
        paidDate: {
          [Op.gte]: new Date(year, 0, 1),
          [Op.lt]: new Date(year + 1, 0, 1)
        }
      }
    });

    const paymentStatusOverview = await Payment.findAll({
      attributes: [
        'status',
        [Payment.sequelize.fn('COUNT', Payment.sequelize.col('id')), 'count'],
        [Payment.sequelize.fn('SUM', Payment.sequelize.col('amount')), 'total']
      ],
      group: ['status']
    });

    return {
      totalRevenue: totalRevenue || 0,
      totalPayments,
      paymentStatusOverview
    };
  },

  // Generate payment report
  generatePaymentReport: async (reportCriteria, userId, userRole) => {
    try {
      const { startDate, endDate, landlordId, propertyId, format = 'json' } = reportCriteria;

      // Check if user can generate reports
      const canGenerateReports = paymentService.canGenerateReports(userRole);
      if (!canGenerateReports) {
        throw new Error('Not authorized to generate payment reports');
      }

      const where = {};
      
      if (startDate && endDate) {
        where.paidDate = {
          [Op.between]: [new Date(startDate), new Date(endDate)]
        };
      }

      if (landlordId) where.landlordId = landlordId;
      if (propertyId) where.propertyId = propertyId;

      // For non-admin users, restrict to their own data
      if (!userRole.includes('admin') && userRole !== 'financial_officer') {
        where.landlordId = userId;
      }

      const payments = await Payment.findAll({
        where: { ...where, status: 'paid' },
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['title', 'address']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['firstName', 'lastName', 'email']
          },
          {
            model: User,
            as: 'landlord',
            attributes: ['firstName', 'lastName', 'email', 'companyName']
          }
        ],
        order: [['paidDate', 'DESC']]
      });

      // Log report generation
      await AuditLog.create({
        action: 'payment_report_generated',
        entityType: 'report',
        userId: userId,
        description: `Payment report generated for period ${startDate} to ${endDate}`,
        metadata: {
          reportType: 'payment',
          startDate,
          endDate,
          recordCount: payments.length
        }
      });

      if (format === 'csv') {
        // Generate CSV data
        const csvData = payments.map(payment => ({
          Date: payment.paidDate?.toISOString().split('T')[0] || '',
          Property: payment.property.title,
          Tenant: `${payment.tenant.firstName} ${payment.tenant.lastName}`,
          Landlord: payment.landlord.companyName || `${payment.landlord.firstName} ${payment.landlord.lastName}`,
          Amount: payment.amount,
          Method: payment.paymentMethod || 'N/A',
          TransactionID: payment.transactionId || 'N/A'
        }));

        return {
          success: true,
          data: {
            format: 'csv',
            data: csvData,
            filename: `payments-report-${moment().format('YYYY-MM-DD')}.csv`
          }
        };
      }

      return {
        success: true,
        data: {
          format: 'json',
          payments,
          summary: {
            totalAmount: payments.reduce((sum, payment) => sum + parseFloat(payment.amount), 0),
            totalPayments: payments.length,
            period: { startDate, endDate }
          }
        }
      };
    } catch (error) {
      console.error('Payment service - generate report error:', error);
      throw error;
    }
  },

  // Create payment schedule for lease
  createPaymentSchedule: async (lease) => {
    try {
      const payments = [];
      const startDate = new Date(lease.startDate);
      const endDate = new Date(lease.endDate);
      const monthlyRent = lease.monthlyRent;

      let currentDate = new Date(startDate);

      while (currentDate < endDate) {
        const dueDate = new Date(currentDate);
        dueDate.setMonth(dueDate.getMonth() + 1);
        
        if (dueDate > endDate) break;

        payments.push({
          amount: monthlyRent,
          dueDate: dueDate,
          status: 'pending',
          leaseId: lease.id,
          tenantId: lease.tenantId,
          landlordId: lease.landlordId,
          propertyId: lease.propertyId,
          description: `Rent for ${dueDate.toLocaleDateString('en-US', { month: 'long', year: 'numeric' })}`
        });

        currentDate = new Date(dueDate);
      }

      if (payments.length > 0) {
        await Payment.bulkCreate(payments);
      }

      return {
        success: true,
        data: { paymentsCreated: payments.length }
      };
    } catch (error) {
      console.error('Payment service - create payment schedule error:', error);
      throw error;
    }
  },

  // Send payment reminders
  sendPaymentReminders: async () => {
    try {
      const overduePayments = await Payment.findAll({
        where: {
          status: 'pending',
          dueDate: { [Op.lt]: new Date() }
        },
        include: ['tenant', 'property', 'landlord']
      });

      const reminderResults = [];

      for (const payment of overduePayments) {
        try {
          await emailService.sendPaymentReminder(payment.tenant, payment, payment.property);
          reminderResults.push({
            paymentId: payment.id,
            tenant: payment.tenant.email,
            status: 'sent'
          });
        } catch (emailError) {
          console.error(`Failed to send reminder for payment ${payment.id}:`, emailError);
          reminderResults.push({
            paymentId: payment.id,
            tenant: payment.tenant.email,
            status: 'failed',
            error: emailError.message
          });
        }
      }

      return {
        success: true,
        data: {
          remindersSent: reminderResults.filter(r => r.status === 'sent').length,
          remindersFailed: reminderResults.filter(r => r.status === 'failed').length,
          results: reminderResults
        }
      };
    } catch (error) {
      console.error('Payment service - send payment reminders error:', error);
      throw error;
    }
  },

  // Permission check methods
  canViewPayment: async (payment, userId, userRole) => {
    if (userRole.includes('admin')) {
      return true;
    }

    if (payment.tenantId === userId || payment.landlordId === userId) {
      return true;
    }

    const staffRoles = [
      'property_manager',
      'financial_officer',
      'financial_analyst',
      'leasing_consultant',
      'customer_support_agent'
    ];

    return staffRoles.includes(userRole);
  },

  canProcessPayment: async (payment, userId, userRole) => {
    if (payment.tenantId === userId) {
      return true;
    }

    if (userRole.includes('admin')) {
      return true;
    }

    const allowedRoles = [
      'financial_officer',
      'property_manager',
      'leasing_consultant'
    ];

    return allowedRoles.includes(userRole);
  },

  canRecordManualPayment: async (payment, userId, userRole) => {
    if (payment.landlordId === userId) {
      return true;
    }

    if (userRole.includes('admin')) {
      return true;
    }

    const allowedRoles = [
      'financial_officer',
      'property_manager'
    ];

    return allowedRoles.includes(userRole);
  },

  canGenerateReports: (userRole) => {
    const allowedRoles = [
      'super_admin',
      'system_admin',
      'landlord',
      'financial_officer',
      'financial_analyst',
      'property_manager',
      'data_analyst'
    ];

    return allowedRoles.includes(userRole);
  }
};

module.exports = paymentService;