const pool = require('../config/database');
const bcrypt = require('bcryptjs');

const seedDatabase = async () => {
  try {
    console.log('🌱 Seeding unified database...');

    // Create sample users
    const hashedPassword = await bcrypt.hash('password123', 10);
    
    const users = [
      {
        first_name: 'John',
        last_name: 'Doe',
        email: 'landlord@stayspot.co.ke',
        password_hash: hashedPassword,
        role: 'landlord',
        phone: '+254712345678',
        is_active: true,
        email_verified: true
      },
      {
        first_name: 'Jane',
        last_name: 'Smith',
        email: 'tenant@stayspot.co.ke',
        password_hash: hashedPassword,
        role: 'tenant',
        phone: '+254723456789',
        is_active: true,
        email_verified: true
      },
      {
        first_name: 'Mike',
        last_name: 'Johnson',
        email: 'agent@stayspot.co.ke',
        password_hash: hashedPassword,
        role: 'agent',
        phone: '+254734567890',
        is_active: true,
        email_verified: true
      },
      {
        first_name: 'Sarah',
        last_name: 'Wilson',
        email: 'manager@stayspot.co.ke',
        password_hash: hashedPassword,
        role: 'property_manager',
        phone: '+254745678901',
        is_active: true,
        email_verified: true
      },
      {
        first_name: 'David',
        last_name: 'Brown',
        email: 'driver@stayspot.co.ke',
        password_hash: hashedPassword,
        role: 'driver',
        phone: '+254756789012',
        is_active: true,
        email_verified: true
      }
    ];

    // Insert users
    for (const user of users) {
      await pool.execute(
        `INSERT INTO users (first_name, last_name, email, password_hash, role, phone, is_active, email_verified) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
        [user.first_name, user.last_name, user.email, user.password_hash, user.role, user.phone, user.is_active, user.email_verified]
      );
    }

    // Create sample properties
    const properties = [
      {
        owner_id: 1, // John Doe (landlord)
        title: 'Modern 2BR Apartment in Nakuru CBD',
        description: 'Beautiful modern apartment with city views, fully furnished with modern amenities.',
        address: '123 Kenyatta Avenue, Nakuru CBD',
        city: 'Nakuru',
        state: 'Nakuru County',
        zip_code: '20100',
        property_type: 'apartment',
        bedrooms: 2,
        bathrooms: 2,
        square_feet: 1200,
        rent_amount: 45000.00,
        deposit_amount: 90000.00,
        is_available: true,
        images: JSON.stringify(['https://images.unsplash.com/photo-1560518883-ce09059eeffa']),
        amenities: JSON.stringify(['parking', 'wifi', 'security', 'water', 'electricity'])
      },
      {
        owner_id: 1,
        title: 'Spacious 3BR House in Milimani',
        description: 'Family-friendly house in quiet neighborhood with garden and parking.',
        address: '456 Milimani Estate, Nakuru',
        city: 'Nakuru',
        state: 'Nakuru County',
        zip_code: '20100',
        property_type: 'house',
        bedrooms: 3,
        bathrooms: 2.5,
        square_feet: 1800,
        rent_amount: 65000.00,
        deposit_amount: 130000.00,
        is_available: true,
        images: JSON.stringify(['https://images.unsplash.com/photo-1486406146926-c627a92ad1ab']),
        amenities: JSON.stringify(['garden', 'parking', 'security', 'water', 'electricity', 'backup_generator'])
      }
    ];

    for (const property of properties) {
      await pool.execute(
        `INSERT INTO properties (owner_id, title, description, address, city, state, zip_code, property_type, bedrooms, bathrooms, square_feet, rent_amount, deposit_amount, is_available, images, amenities) 
         VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)`,
        [property.owner_id, property.title, property.description, property.address, property.city, property.state, property.zip_code, property.property_type, property.bedrooms, property.bathrooms, property.square_feet, property.rent_amount, property.deposit_amount, property.is_available, property.images, property.amenities]
      );
    }

    // Create sample lease
    await pool.execute(
      `INSERT INTO leases (property_id, tenant_id, landlord_id, start_date, end_date, monthly_rent, security_deposit, status) 
       VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
      [1, 2, 1, '2024-01-01', '2024-12-31', 45000.00, 90000.00, 'active']
    );

    // Create sample payment
    await pool.execute(
      `INSERT INTO payments (lease_id, tenant_id, amount, payment_type, payment_method, payment_date, due_date, status) 
       VALUES (?, ?, ?, ?, ?, ?, ?, ?)`,
      [1, 2, 45000.00, 'rent', 'bank_transfer', '2024-01-01', '2024-01-01', 'completed']
    );

    // Create sample maintenance request
    await pool.execute(
      `INSERT INTO maintenance_requests (property_id, tenant_id, title, description, priority, category, status) 
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [1, 2, 'Leaking Faucet', 'Kitchen faucet is leaking and needs repair', 'medium', 'plumbing', 'pending']
    );

    // Create sample notifications
    const notifications = [
      {
        user_id: 1,
        title: 'New Tenant Application',
        message: 'You have received a new tenant application for your property.',
        type: 'info'
      },
      {
        user_id: 2,
        title: 'Rent Due Reminder',
        message: 'Your rent payment is due in 3 days.',
        type: 'payment'
      }
    ];

    for (const notification of notifications) {
      await pool.execute(
        `INSERT INTO notifications (user_id, title, message, type) VALUES (?, ?, ?, ?)`,
        [notification.user_id, notification.title, notification.message, notification.type]
      );
    }

    // Create sample review
    await pool.execute(
      `INSERT INTO reviews (rating, title, comment, review_type, reviewer_id, property_id, is_published) 
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [5, 'Excellent Property!', 'Great location and well-maintained property. Highly recommend!', 'property', 2, 1, true]
    );

    // Create sample document
    await pool.execute(
      `INSERT INTO documents (title, description, file_url, file_name, document_type, uploaded_by, lease_id) 
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
      ['Lease Agreement', 'Signed lease agreement for property rental', '/uploads/lease_agreement_001.pdf', 'lease_agreement_001.pdf', 'lease_agreement', 1, 1]
    );

    // Create sample message
    await pool.execute(
      `INSERT INTO messages (content, sender_id, receiver_id, conversation_id, property_id) 
       VALUES (?, ?, ?, ?, ?)`,
      ['Hello, I have a question about the property maintenance.', 2, 1, 'conv_1_2_prop_1', 1]
    );

    // Create sample inspection
    await pool.execute(
      `INSERT INTO inspections (property_id, inspector_id, inspection_type, scheduled_date, status) 
       VALUES (?, ?, ?, ?, ?)`,
      [1, 3, 'routine', '2024-02-15', 'scheduled']
    );

    // Create sample expense
    await pool.execute(
      `INSERT INTO expenses (property_id, category, description, amount, expense_date, created_by, status) 
       VALUES (?, ?, ?, ?, ?, ?, ?)`,
      [1, 'maintenance', 'Plumbing repair for kitchen sink', 2500.00, '2024-01-15', 1, 'approved']
    );

    // Create sample application
    await pool.execute(
      `INSERT INTO applications (property_id, applicant_id, status, move_in_date, monthly_income) 
       VALUES (?, ?, ?, ?, ?)`,
      [2, 2, 'pending', '2024-03-01', 75000.00]
    );

    // Create sample vendor
    await pool.execute(
      `INSERT INTO vendors (company_name, contact_person, email, phone, service_category, rating) 
       VALUES (?, ?, ?, ?, ?, ?)`,
      ['Nakuru Plumbing Services', 'James Mwangi', 'james@nakuruplumbing.co.ke', '+254722123456', 'plumbing', 4.5]
    );

    console.log('✅ Database seeded successfully!');
    console.log('👥 Created 5 sample users');
    console.log('🏠 Created 2 sample properties');
    console.log('📄 Created 1 sample lease');
    console.log('💰 Created 1 sample payment');
    console.log('🔧 Created 1 sample maintenance request');
    console.log('🔔 Created 2 sample notifications');
    console.log('⭐ Created 1 sample review');
    console.log('📋 Created 1 sample document');
    console.log('💬 Created 1 sample message');
    console.log('🔍 Created 1 sample inspection');
    console.log('💸 Created 1 sample expense');
    console.log('📝 Created 1 sample application');
    console.log('🔧 Created 1 sample vendor');
    console.log('');
    console.log('Sample login credentials:');
    console.log('Landlord: landlord@stayspot.co.ke / password123');
    console.log('Tenant: tenant@stayspot.co.ke / password123');
    console.log('Agent: agent@stayspot.co.ke / password123');
    console.log('Manager: manager@stayspot.co.ke / password123');
    console.log('Driver: driver@stayspot.co.ke / password123');

  } catch (error) {
    console.error('❌ Error seeding database:', error.message);
    throw error;
  }
};

// Run if called directly
if (require.main === module) {
  require('dotenv').config();
  seedDatabase()
    .then(() => {
      console.log('Database seeding completed successfully');
      process.exit(0);
    })
    .catch((error) => {
      console.error('Database seeding failed:', error);
      process.exit(1);
    });
}

module.exports = seedDatabase;