-- StaySpot Unified Database Schema
-- All tables in one database for better management

CREATE DATABASE IF NOT EXISTS stayspotco_stayspot;
USE stayspotco_stayspot;

-- Users table (central user management)
CREATE TABLE users (
    id INT PRIMARY KEY AUTO_INCREMENT,
    first_name VARCHAR(100) NOT NULL,
    last_name VARCHAR(100) NOT NULL,
    email VARCHAR(255) UNIQUE NOT NULL,
    password_hash VARCHAR(255) NOT NULL,
    role ENUM('landlord', 'tenant', 'agent', 'property_manager', 'maintenance_staff', 'maintenance_supervisor', 'leasing_consultant', 'financial_analyst', 'financial_officer', 'marketing_specialist', 'legal_advisor', 'inspector', 'insurance_coordinator', 'customer_support_agent', 'data_analyst', 'vendor', 'bnb_host', 'community_manager', 'regional_manager', 'relocation_specialist', 'driver') DEFAULT 'tenant',
    phone VARCHAR(20),
    profile_image VARCHAR(255),
    is_active BOOLEAN DEFAULT true,
    email_verified BOOLEAN DEFAULT false,
    last_login TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_role (role),
    INDEX idx_active (is_active)
);

-- Properties table
CREATE TABLE properties (
    id INT PRIMARY KEY AUTO_INCREMENT,
    owner_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    address TEXT NOT NULL,
    city VARCHAR(100) NOT NULL,
    state VARCHAR(100) NOT NULL,
    zip_code VARCHAR(20),
    country VARCHAR(100) DEFAULT 'Kenya',
    property_type ENUM('apartment', 'house', 'condo', 'townhouse', 'studio', 'commercial', 'office', 'warehouse') NOT NULL,
    bedrooms INT DEFAULT 0,
    bathrooms DECIMAL(3,1) DEFAULT 0,
    square_feet INT,
    rent_amount DECIMAL(10,2) NOT NULL,
    deposit_amount DECIMAL(10,2),
    is_available BOOLEAN DEFAULT true,
    images JSON,
    amenities JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (owner_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_owner (owner_id),
    INDEX idx_available (is_available),
    INDEX idx_city (city),
    INDEX idx_type (property_type)
);

-- Leases table
CREATE TABLE leases (
    id INT PRIMARY KEY AUTO_INCREMENT,
    property_id INT NOT NULL,
    tenant_id INT NOT NULL,
    landlord_id INT NOT NULL,
    start_date DATE NOT NULL,
    end_date DATE NOT NULL,
    monthly_rent DECIMAL(10,2) NOT NULL,
    security_deposit DECIMAL(10,2),
    status ENUM('active', 'expired', 'terminated', 'pending') DEFAULT 'pending',
    lease_terms TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (tenant_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (landlord_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_property (property_id),
    INDEX idx_tenant (tenant_id),
    INDEX idx_landlord (landlord_id),
    INDEX idx_status (status)
);

-- Payments table
CREATE TABLE payments (
    id INT PRIMARY KEY AUTO_INCREMENT,
    lease_id INT NOT NULL,
    tenant_id INT NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    payment_type ENUM('rent', 'deposit', 'fee', 'utility', 'maintenance', 'late_fee') NOT NULL,
    payment_method ENUM('cash', 'check', 'bank_transfer', 'credit_card', 'mobile_money') NOT NULL,
    payment_date DATE NOT NULL,
    due_date DATE NOT NULL,
    status ENUM('pending', 'completed', 'failed', 'refunded') DEFAULT 'pending',
    transaction_id VARCHAR(255),
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (lease_id) REFERENCES leases(id) ON DELETE CASCADE,
    FOREIGN KEY (tenant_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_lease (lease_id),
    INDEX idx_tenant (tenant_id),
    INDEX idx_status (status),
    INDEX idx_payment_date (payment_date)
);

-- Maintenance requests table
CREATE TABLE maintenance_requests (
    id INT PRIMARY KEY AUTO_INCREMENT,
    property_id INT NOT NULL,
    tenant_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    description TEXT NOT NULL,
    priority ENUM('low', 'medium', 'high', 'urgent') DEFAULT 'medium',
    status ENUM('pending', 'in_progress', 'completed', 'cancelled') DEFAULT 'pending',
    category ENUM('plumbing', 'electrical', 'hvac', 'appliance', 'structural', 'pest_control', 'cleaning', 'other') NOT NULL,
    assigned_to INT NULL,
    estimated_cost DECIMAL(10,2),
    actual_cost DECIMAL(10,2),
    scheduled_date DATE,
    completed_date DATE,
    images JSON,
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (tenant_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (assigned_to) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_property (property_id),
    INDEX idx_tenant (tenant_id),
    INDEX idx_status (status),
    INDEX idx_priority (priority)
);

-- Notifications table
CREATE TABLE notifications (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    title VARCHAR(255) NOT NULL,
    message TEXT NOT NULL,
    type ENUM('info', 'success', 'warning', 'error', 'payment', 'maintenance', 'lease') DEFAULT 'info',
    is_read BOOLEAN DEFAULT false,
    action_url VARCHAR(255),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    read_at TIMESTAMP NULL,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_user (user_id),
    INDEX idx_read (is_read),
    INDEX idx_type (type)
);

-- Chat messages table
CREATE TABLE chat_messages (
    id INT PRIMARY KEY AUTO_INCREMENT,
    room_id VARCHAR(100) NOT NULL,
    sender_id INT NOT NULL,
    message TEXT NOT NULL,
    message_type ENUM('text', 'image', 'file') DEFAULT 'text',
    file_url VARCHAR(255),
    is_read BOOLEAN DEFAULT false,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (sender_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_room (room_id),
    INDEX idx_sender (sender_id),
    INDEX idx_created (created_at)
);

-- Partnership applications table
CREATE TABLE partnership_applications (
    id INT PRIMARY KEY AUTO_INCREMENT,
    company_name VARCHAR(255) NOT NULL,
    contact_person VARCHAR(255) NOT NULL,
    email VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    partnership_type ENUM('vendor', 'service_provider', 'technology', 'marketing', 'financial') NOT NULL,
    description TEXT NOT NULL,
    website VARCHAR(255),
    status ENUM('pending', 'approved', 'rejected', 'under_review') DEFAULT 'pending',
    notes TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_status (status),
    INDEX idx_type (partnership_type)
);

-- Transportation/Moving services table
CREATE TABLE transportation_bookings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NOT NULL,
    driver_id INT NULL,
    service_type ENUM('moving', 'delivery', 'pickup') NOT NULL,
    pickup_address TEXT NOT NULL,
    delivery_address TEXT NOT NULL,
    pickup_date DATE NOT NULL,
    pickup_time TIME NOT NULL,
    estimated_cost DECIMAL(10,2),
    actual_cost DECIMAL(10,2),
    status ENUM('pending', 'confirmed', 'in_progress', 'completed', 'cancelled') DEFAULT 'pending',
    vehicle_type ENUM('truck', 'van', 'pickup', 'motorcycle') NOT NULL,
    items_description TEXT,
    special_instructions TEXT,
    tracking_code VARCHAR(50) UNIQUE,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (driver_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user (user_id),
    INDEX idx_driver (driver_id),
    INDEX idx_status (status),
    INDEX idx_tracking (tracking_code)
);

-- Analytics/Reports table
CREATE TABLE analytics_data (
    id INT PRIMARY KEY AUTO_INCREMENT,
    metric_name VARCHAR(100) NOT NULL,
    metric_value DECIMAL(15,2) NOT NULL,
    metric_type ENUM('revenue', 'occupancy', 'maintenance_cost', 'user_count', 'property_count') NOT NULL,
    period_type ENUM('daily', 'weekly', 'monthly', 'yearly') NOT NULL,
    period_date DATE NOT NULL,
    metadata JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    INDEX idx_metric (metric_name),
    INDEX idx_type (metric_type),
    INDEX idx_period (period_date)
);

-- Audit logs table
CREATE TABLE audit_logs (
    id INT PRIMARY KEY AUTO_INCREMENT,
    user_id INT NULL,
    action VARCHAR(100) NOT NULL,
    table_name VARCHAR(100) NOT NULL,
    record_id INT NOT NULL,
    old_values JSON,
    new_values JSON,
    ip_address VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_user (user_id),
    INDEX idx_action (action),
    INDEX idx_table (table_name),
    INDEX idx_created (created_at)
);

-- Reviews table
CREATE TABLE reviews (
    id INT PRIMARY KEY AUTO_INCREMENT,
    rating INT NOT NULL CHECK (rating >= 1 AND rating <= 5),
    title VARCHAR(255),
    comment TEXT,
    review_type ENUM('property', 'landlord', 'tenant') NOT NULL,
    response TEXT,
    responded_at TIMESTAMP NULL,
    is_published BOOLEAN DEFAULT true,
    helpful_count INT DEFAULT 0,
    reviewer_id INT NOT NULL,
    reviewee_id INT NULL,
    property_id INT NULL,
    lease_id INT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (reviewer_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewee_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (lease_id) REFERENCES leases(id) ON DELETE CASCADE,
    INDEX idx_review_type (review_type),
    INDEX idx_property_reviews (property_id, is_published),
    INDEX idx_rating (rating)
);

-- Documents table
CREATE TABLE documents (
    id INT PRIMARY KEY AUTO_INCREMENT,
    title VARCHAR(255) NOT NULL,
    description TEXT,
    file_url VARCHAR(500) NOT NULL,
    file_name VARCHAR(255) NOT NULL,
    file_size INT,
    mime_type VARCHAR(100),
    document_type ENUM('lease_agreement', 'id_proof', 'income_proof', 'property_document', 'maintenance_report', 'inspection_report', 'insurance_document', 'other') NOT NULL,
    category VARCHAR(100),
    tags JSON,
    is_public BOOLEAN DEFAULT false,
    expires_at DATE,
    uploaded_by INT NOT NULL,
    property_id INT NULL,
    lease_id INT NULL,
    user_id INT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (uploaded_by) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (lease_id) REFERENCES leases(id) ON DELETE CASCADE,
    FOREIGN KEY (user_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_document_type (document_type),
    INDEX idx_property_docs (property_id),
    INDEX idx_lease_docs (lease_id)
);

-- Messages table (for direct messaging)
CREATE TABLE messages (
    id INT PRIMARY KEY AUTO_INCREMENT,
    content TEXT NOT NULL,
    message_type ENUM('text', 'image', 'file', 'system') DEFAULT 'text',
    attachments JSON,
    is_read BOOLEAN DEFAULT false,
    read_at TIMESTAMP NULL,
    sender_id INT NOT NULL,
    receiver_id INT NOT NULL,
    property_id INT NULL,
    lease_id INT NULL,
    parent_message_id INT NULL,
    conversation_id VARCHAR(100) NOT NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (sender_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (receiver_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (lease_id) REFERENCES leases(id) ON DELETE CASCADE,
    FOREIGN KEY (parent_message_id) REFERENCES messages(id) ON DELETE CASCADE,
    INDEX idx_conversation (conversation_id),
    INDEX idx_sender_receiver (sender_id, receiver_id),
    INDEX idx_created (created_at)
);

-- Inspections table
CREATE TABLE inspections (
    id INT PRIMARY KEY AUTO_INCREMENT,
    property_id INT NOT NULL,
    inspector_id INT NOT NULL,
    inspection_type ENUM('move_in', 'move_out', 'routine', 'maintenance', 'safety') NOT NULL,
    scheduled_date DATE NOT NULL,
    completed_date DATE NULL,
    status ENUM('scheduled', 'in_progress', 'completed', 'cancelled') DEFAULT 'scheduled',
    checklist JSON,
    findings TEXT,
    recommendations TEXT,
    photos JSON,
    overall_condition ENUM('excellent', 'good', 'fair', 'poor') NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (inspector_id) REFERENCES users(id) ON DELETE CASCADE,
    INDEX idx_property_inspections (property_id),
    INDEX idx_inspector (inspector_id),
    INDEX idx_status (status)
);

-- Expenses table
CREATE TABLE expenses (
    id INT PRIMARY KEY AUTO_INCREMENT,
    property_id INT NULL,
    category ENUM('maintenance', 'utilities', 'insurance', 'taxes', 'management', 'marketing', 'legal', 'other') NOT NULL,
    description TEXT NOT NULL,
    amount DECIMAL(10,2) NOT NULL,
    expense_date DATE NOT NULL,
    vendor VARCHAR(255),
    receipt_url VARCHAR(500),
    is_recurring BOOLEAN DEFAULT false,
    recurring_frequency ENUM('monthly', 'quarterly', 'yearly') NULL,
    created_by INT NOT NULL,
    approved_by INT NULL,
    status ENUM('pending', 'approved', 'rejected') DEFAULT 'pending',
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (created_by) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (approved_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_property_expenses (property_id),
    INDEX idx_category (category),
    INDEX idx_expense_date (expense_date)
);

-- Applications table (tenant applications)
CREATE TABLE applications (
    id INT PRIMARY KEY AUTO_INCREMENT,
    property_id INT NOT NULL,
    applicant_id INT NOT NULL,
    application_type ENUM('rental', 'lease_renewal', 'transfer') DEFAULT 'rental',
    status ENUM('pending', 'approved', 'rejected', 'withdrawn') DEFAULT 'pending',
    move_in_date DATE,
    employment_info JSON,
    references JSON,
    background_check_status ENUM('pending', 'passed', 'failed') NULL,
    credit_score INT NULL,
    monthly_income DECIMAL(10,2),
    notes TEXT,
    reviewed_by INT NULL,
    reviewed_at TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (property_id) REFERENCES properties(id) ON DELETE CASCADE,
    FOREIGN KEY (applicant_id) REFERENCES users(id) ON DELETE CASCADE,
    FOREIGN KEY (reviewed_by) REFERENCES users(id) ON DELETE SET NULL,
    INDEX idx_property_applications (property_id),
    INDEX idx_applicant (applicant_id),
    INDEX idx_status (status)
);

-- Vendors table
CREATE TABLE vendors (
    id INT PRIMARY KEY AUTO_INCREMENT,
    company_name VARCHAR(255) NOT NULL,
    contact_person VARCHAR(255) NOT NULL,
    email VARCHAR(255) NOT NULL,
    phone VARCHAR(20),
    address TEXT,
    service_category ENUM('plumbing', 'electrical', 'hvac', 'cleaning', 'landscaping', 'security', 'general_maintenance', 'other') NOT NULL,
    rating DECIMAL(3,2) DEFAULT 0.00,
    is_active BOOLEAN DEFAULT true,
    license_number VARCHAR(100),
    insurance_info JSON,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_category (service_category),
    INDEX idx_active (is_active),
    INDEX idx_rating (rating)
);

-- Settings table
CREATE TABLE system_settings (
    id INT PRIMARY KEY AUTO_INCREMENT,
    setting_key VARCHAR(100) UNIQUE NOT NULL,
    setting_value TEXT NOT NULL,
    setting_type ENUM('string', 'number', 'boolean', 'json') DEFAULT 'string',
    description TEXT,
    is_public BOOLEAN DEFAULT false,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_key (setting_key)
);

-- Insert default system settings
INSERT INTO system_settings (setting_key, setting_value, setting_type, description, is_public) VALUES
('app_name', 'StaySpot', 'string', 'Application name', true),
('app_version', '1.0.0', 'string', 'Application version', true),
('maintenance_mode', 'false', 'boolean', 'Enable maintenance mode', false),
('max_file_size', '10485760', 'number', 'Maximum file upload size in bytes', false),
('default_currency', 'KES', 'string', 'Default currency code', true),
('company_phone', '+254748041595', 'string', 'Company phone number', true),
('company_email', 'info@stayspot.co.ke', 'string', 'Company email address', true),
('company_address', 'Nakuru, Kenya', 'string', 'Company address', true);

-- Create indexes for better performance
CREATE INDEX idx_users_email_active ON users(email, is_active);
CREATE INDEX idx_properties_owner_available ON properties(owner_id, is_available);
CREATE INDEX idx_leases_dates ON leases(start_date, end_date);
CREATE INDEX idx_payments_dates ON payments(payment_date, due_date);
CREATE INDEX idx_maintenance_priority_status ON maintenance_requests(priority, status);

-- Create views for common queries
CREATE VIEW active_leases AS
SELECT 
    l.*,
    p.title as property_title,
    p.address as property_address,
    CONCAT(t.first_name, ' ', t.last_name) as tenant_name,
    t.email as tenant_email,
    CONCAT(ll.first_name, ' ', ll.last_name) as landlord_name
FROM leases l
JOIN properties p ON l.property_id = p.id
JOIN users t ON l.tenant_id = t.id
JOIN users ll ON l.landlord_id = ll.id
WHERE l.status = 'active';

CREATE VIEW property_summary AS
SELECT 
    p.*,
    CONCAT(u.first_name, ' ', u.last_name) as owner_name,
    u.email as owner_email,
    COUNT(l.id) as total_leases,
    SUM(CASE WHEN l.status = 'active' THEN 1 ELSE 0 END) as active_leases
FROM properties p
JOIN users u ON p.owner_id = u.id
LEFT JOIN leases l ON p.id = l.property_id
GROUP BY p.id;

CREATE VIEW payment_summary AS
SELECT 
    p.*,
    CONCAT(u.first_name, ' ', u.last_name) as tenant_name,
    pr.title as property_title
FROM payments p
JOIN users u ON p.tenant_id = u.id
JOIN leases l ON p.lease_id = l.id
JOIN properties pr ON l.property_id = pr.id;

CREATE VIEW maintenance_summary AS
SELECT 
    m.*,
    p.title as property_title,
    p.address as property_address,
    CONCAT(t.first_name, ' ', t.last_name) as tenant_name,
    CONCAT(a.first_name, ' ', a.last_name) as assigned_to_name
FROM maintenance_requests m
JOIN properties p ON m.property_id = p.id
JOIN users t ON m.tenant_id = t.id
LEFT JOIN users a ON m.assigned_to = a.id;

CREATE VIEW property_analytics AS
SELECT 
    p.id,
    p.title,
    p.rent_amount,
    COUNT(DISTINCT l.id) as total_leases,
    COUNT(DISTINCT CASE WHEN l.status = 'active' THEN l.id END) as active_leases,
    COUNT(DISTINCT m.id) as maintenance_requests,
    COUNT(DISTINCT r.id) as reviews,
    AVG(r.rating) as average_rating,
    SUM(pay.amount) as total_revenue
FROM properties p
LEFT JOIN leases l ON p.id = l.property_id
LEFT JOIN maintenance_requests m ON p.id = m.property_id
LEFT JOIN reviews r ON p.id = r.property_id AND r.review_type = 'property'
LEFT JOIN payments pay ON l.id = pay.lease_id AND pay.status = 'completed'
GROUP BY p.id;

CREATE VIEW user_dashboard AS
SELECT 
    u.id,
    u.first_name,
    u.last_name,
    u.email,
    u.role,
    COUNT(DISTINCT CASE WHEN u.role = 'landlord' THEN p.id END) as owned_properties,
    COUNT(DISTINCT CASE WHEN u.role = 'tenant' THEN l.id END) as active_leases,
    COUNT(DISTINCT n.id) as unread_notifications,
    COUNT(DISTINCT m.id) as unread_messages
FROM users u
LEFT JOIN properties p ON u.id = p.owner_id
LEFT JOIN leases l ON u.id = l.tenant_id AND l.status = 'active'
LEFT JOIN notifications n ON u.id = n.user_id AND n.is_read = false
LEFT JOIN messages m ON u.id = m.receiver_id AND m.is_read = false
GROUP BY u.id;