const { Notification, AuditLog, Payment, Maintenance } = require('../models');
const { Op } = require('sequelize');

const cleanupScript = {
  // Clean up expired notifications
  cleanupExpiredNotifications: async () => {
    try {
      const result = await Notification.destroy({
        where: {
          expiresAt: {
            [Op.lt]: new Date()
          }
        }
      });

      console.log(`Cleaned up ${result} expired notifications`);
      return result;
    } catch (error) {
      console.error('Error cleaning up expired notifications:', error);
      throw error;
    }
  },

  // Clean up old audit logs
  cleanupOldAuditLogs: async (retentionDays = 90) => {
    try {
      const cutoffDate = new Date();
      cutoffDate.setDate(cutoffDate.getDate() - retentionDays);

      const result = await AuditLog.destroy({
        where: {
          createdAt: {
            [Op.lt]: cutoffDate
          }
        }
      });

      console.log(`Cleaned up ${result} audit logs older than ${retentionDays} days`);
      return result;
    } catch (error) {
      console.error('Error cleaning up old audit logs:', error);
      throw error;
    }
  },

  // Clean up failed payments
  cleanupFailedPayments: async (retentionDays = 30) => {
    try {
      const cutoffDate = new Date();
      cutoffDate.setDate(cutoffDate.getDate() - retentionDays);

      const result = await Payment.destroy({
        where: {
          status: 'failed',
          createdAt: {
            [Op.lt]: cutoffDate
          }
        }
      });

      console.log(`Cleaned up ${result} failed payments older than ${retentionDays} days`);
      return result;
    } catch (error) {
      console.error('Error cleaning up failed payments:', error);
      throw error;
    }
  },

  // Clean up completed maintenance requests
  cleanupCompletedMaintenance: async (retentionDays = 180) => {
    try {
      const cutoffDate = new Date();
      cutoffDate.setDate(cutoffDate.getDate() - retentionDays);

      const result = await Maintenance.destroy({
        where: {
          status: 'completed',
          completedDate: {
            [Op.lt]: cutoffDate
          }
        }
      });

      console.log(`Cleaned up ${result} completed maintenance requests older than ${retentionDays} days`);
      return result;
    } catch (error) {
      console.error('Error cleaning up completed maintenance requests:', error);
      throw error;
    }
  },

  // Clean up soft-deleted records
  cleanupSoftDeletedRecords: async () => {
    try {
      // This would require adding paranoid deletion to models
      // For now, it's a placeholder for future implementation
      console.log('Soft-deleted records cleanup not implemented yet');
      return 0;
    } catch (error) {
      console.error('Error cleaning up soft-deleted records:', error);
      throw error;
    }
  },

  // Run all cleanup tasks
  runAllCleanup: async () => {
    try {
      console.log('Starting comprehensive cleanup...');

      const results = {
        expiredNotifications: await cleanupScript.cleanupExpiredNotifications(),
        oldAuditLogs: await cleanupScript.cleanupOldAuditLogs(),
        failedPayments: await cleanupScript.cleanupFailedPayments(),
        completedMaintenance: await cleanupScript.cleanupCompletedMaintenance()
      };

      console.log('Cleanup completed successfully:');
      console.log(JSON.stringify(results, null, 2));

      return results;
    } catch (error) {
      console.error('Comprehensive cleanup failed:', error);
      throw error;
    }
  }
};

// Run cleanup if called directly
if (require.main === module) {
  const command = process.argv[2] || 'all';

  const runCleanup = async () => {
    try {
      switch (command) {
        case 'notifications':
          await cleanupScript.cleanupExpiredNotifications();
          break;
        case 'audit-logs':
          await cleanupScript.cleanupOldAuditLogs();
          break;
        case 'payments':
          await cleanupScript.cleanupFailedPayments();
          break;
        case 'maintenance':
          await cleanupScript.cleanupCompletedMaintenance();
          break;
        case 'all':
        default:
          await cleanupScript.runAllCleanup();
      }

      process.exit(0);
    } catch (error) {
      console.error('Cleanup script failed:', error);
      process.exit(1);
    }
  };

  runCleanup();
}

module.exports = cleanupScript;