const { exec } = require('child_process');
const fs = require('fs');
const path = require('path');
const { sequelize } = require('../config/database');

const backupDatabase = async () => {
  const backupDir = path.join(__dirname, '../backups');
  const timestamp = new Date().toISOString().replace(/[:.]/g, '-');
  const backupFile = path.join(backupDir, `backup-${timestamp}.sql`);

  try {
    // Create backup directory if it doesn't exist
    if (!fs.existsSync(backupDir)) {
      fs.mkdirSync(backupDir, { recursive: true });
    }

    const { DB_NAME, DB_USER, DB_PASS, DB_HOST, DB_PORT } = process.env;

    // Create backup using mysqldump
    const command = `mysqldump -h ${DB_HOST} -P ${DB_PORT} -u ${DB_USER} -p${DB_PASS} ${DB_NAME} > ${backupFile}`;

    exec(command, (error, stdout, stderr) => {
      if (error) {
        console.error('Backup failed:', error);
        return;
      }

      console.log(`Database backup created successfully: ${backupFile}`);

      // Clean up old backups (keep only last 7 days)
      cleanupOldBackups(backupDir);
    });

  } catch (error) {
    console.error('Backup process failed:', error);
  }
};

const cleanupOldBackups = (backupDir) => {
  const retentionDays = 7;
  const cutoffTime = Date.now() - (retentionDays * 24 * 60 * 60 * 1000);

  try {
    const files = fs.readdirSync(backupDir);
    
    files.forEach(file => {
      const filePath = path.join(backupDir, file);
      const stats = fs.statSync(filePath);
      
      if (stats.mtimeMs < cutoffTime && file.startsWith('backup-')) {
        fs.unlinkSync(filePath);
        console.log(`Deleted old backup: ${file}`);
      }
    });
  } catch (error) {
    console.error('Error cleaning up old backups:', error);
  }
};

const listBackups = () => {
  const backupDir = path.join(__dirname, '../backups');
  
  try {
    if (!fs.existsSync(backupDir)) {
      console.log('No backups directory found');
      return;
    }

    const files = fs.readdirSync(backupDir)
      .filter(file => file.startsWith('backup-') && file.endsWith('.sql'))
      .sort()
      .reverse();

    console.log('Available backups:');
    files.forEach((file, index) => {
      const filePath = path.join(backupDir, file);
      const stats = fs.statSync(filePath);
      console.log(`${index + 1}. ${file} (${new Date(stats.mtime).toLocaleString()})`);
    });
  } catch (error) {
    console.error('Error listing backups:', error);
  }
};

// Run backup if called directly
if (require.main === module) {
  const command = process.argv[2];
  
  if (command === 'list') {
    listBackups();
  } else {
    backupDatabase();
  }
}

module.exports = {
  backupDatabase,
  listBackups,
  cleanupOldBackups
};