const express = require('express');
const router = express.Router();
const userController = require('../controllers/userController');
const { authenticateToken } = require('../middleware/auth');
const { authorize } = require('../middleware/roleAuth');
const { validateId } = require('../middleware/validation');
const { handleUpload, uploadToCloudinaryMiddleware, setUploadFolder } = require('../middleware/upload');
const authConfig = require('../config/auth');

// Admin only routes
router.get(
  '/',
  authenticateToken,
  authorize(authConfig.roles.SUPER_ADMIN, authConfig.roles.SYSTEM_ADMIN),
  userController.getAllUsers
);

// Protected routes
router.get(
  '/:id',
  authenticateToken,
  validateId,
  userController.getUserById
);

router.put(
  '/:id',
  authenticateToken,
  validateId,
  userController.updateUser
);

router.delete(
  '/:id',
  authenticateToken,
  authorize(authConfig.roles.SUPER_ADMIN, authConfig.roles.SYSTEM_ADMIN),
  validateId,
  userController.deleteUser
);

router.get(
  '/:id/stats',
  authenticateToken,
  validateId,
  userController.getUserStats
);

router.get(
  '/:id/activity',
  authenticateToken,
  validateId,
  userController.getUserActivity
);

// Avatar upload
router.post(
  '/avatar/upload',
  authenticateToken,
  handleUpload,
  setUploadFolder('avatars'),
  uploadToCloudinaryMiddleware,
  userController.uploadAvatar
);

// Current user routes
router.get(
  '/me/stats',
  authenticateToken,
  (req, res) => userController.getUserStats(req, res)
);

router.get(
  '/me/activity',
  authenticateToken,
  (req, res) => userController.getUserActivity(req, res)
);

module.exports = router;