const express = require('express');
const router = express.Router();
const db = require('../config/database');

// Global search endpoint
router.get('/', async (req, res) => {
  try {
    const { q } = req.query;
    
    if (!q || q.length < 2) {
      return res.json({ results: [] });
    }

    const searchTerm = `%${q}%`;
    const results = [];

    // Search properties
    const propertyQuery = `
      SELECT id, title, location, price, 'property' as type
      FROM properties 
      WHERE title LIKE ? OR location LIKE ? OR description LIKE ?
      LIMIT 5
    `;
    const properties = await db.query(propertyQuery, [searchTerm, searchTerm, searchTerm]);
    
    properties.forEach(property => {
      results.push({
        title: property.title,
        description: `${property.location} - $${property.price}`,
        url: `/properties/${property.id}`,
        icon: 'fa-home',
        type: 'property'
      });
    });

    // Search users (for admin/manager roles)
    const userQuery = `
      SELECT id, name, email, role, 'user' as type
      FROM users 
      WHERE name LIKE ? OR email LIKE ?
      LIMIT 3
    `;
    const users = await db.query(userQuery, [searchTerm, searchTerm]);
    
    users.forEach(user => {
      results.push({
        title: user.name,
        description: `${user.role} - ${user.email}`,
        url: `/dashboard/users/${user.id}`,
        icon: 'fa-user',
        type: 'user'
      });
    });

    // Search maintenance requests
    const maintenanceQuery = `
      SELECT id, title, status, 'maintenance' as type
      FROM maintenance_requests 
      WHERE title LIKE ? OR description LIKE ?
      LIMIT 3
    `;
    const maintenance = await db.query(maintenanceQuery, [searchTerm, searchTerm]);
    
    maintenance.forEach(request => {
      results.push({
        title: request.title,
        description: `Status: ${request.status}`,
        url: `/dashboard/maintenance/${request.id}`,
        icon: 'fa-wrench',
        type: 'maintenance'
      });
    });

    res.json({ results: results.slice(0, 10) });
  } catch (error) {
    console.error('Search error:', error);
    res.status(500).json({ error: 'Search failed' });
  }
});

module.exports = router;