const express = require('express');
const router = express.Router();
const paymentController = require('../controllers/paymentController');
const { authenticateToken } = require('../middleware/auth');
const { authorize } = require('../middleware/roleAuth');
const { validateId } = require('../middleware/validation');
const authConfig = require('../config/auth');

// Get all payments (filtered by user role)
router.get(
  '/',
  authenticateToken,
  paymentController.getAllPayments
);

// Get payment by ID
router.get(
  '/:id',
  authenticateToken,
  validateId,
  paymentController.getPaymentById
);

// Create payment (manual entry for admins/landlords)
router.post(
  '/',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.FINANCIAL_OFFICER
  ),
  paymentController.createPayment
);

// Process payment (tenant making payment)
router.post(
  '/:id/process',
  authenticateToken,
  validateId,
  paymentController.processPayment
);

// Update payment
router.put(
  '/:id',
  authenticateToken,
  validateId,
  paymentController.updatePayment
);

// Record manual payment
router.patch(
  '/:id/record-payment',
  authenticateToken,
  validateId,
  paymentController.recordManualPayment
);

// Refund payment
router.patch(
  '/:id/refund',
  authenticateToken,
  validateId,
  paymentController.refundPayment
);

// Get payment history for tenant
router.get(
  '/tenant/my-payments',
  authenticateToken,
  authorize(authConfig.roles.TENANT),
  (req, res) => {
    req.query.tenantId = req.user.id;
    paymentController.getAllPayments(req, res);
  }
);

// Get payment history for landlord
router.get(
  '/landlord/my-payments',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN
  ),
  (req, res) => {
    req.query.landlordId = req.user.id;
    paymentController.getAllPayments(req, res);
  }
);

// Get overdue payments
router.get(
  '/overdue/list',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.FINANCIAL_OFFICER
  ),
  paymentController.getOverduePayments
);

// Generate payment report
router.get(
  '/reports/financial',
  authenticateToken,
  authorize(
    authConfig.roles.LANDLORD,
    authConfig.roles.SUPER_ADMIN,
    authConfig.roles.SYSTEM_ADMIN,
    authConfig.roles.FINANCIAL_OFFICER
  ),
  paymentController.generateFinancialReport
);

module.exports = router;