import { type Formatter } from '@json2csv/formatters';
import type Transform from './types/Transform.js';
export interface FieldValueGetterInfo<FT> {
    label: string;
    default?: FT;
}
export interface FieldValueGetterFnWithoutField<RT, FT> {
    (row: RT): FT;
}
export interface FieldValueGetterFnWithField<RT, FT> {
    (row: RT, field: FieldValueGetterInfo<FT>): FT;
}
export type FieldValueGetter<RT, FT> = string | FieldValueGetterFnWithoutField<RT, FT> | FieldValueGetterFnWithField<RT, FT>;
export interface FieldInfo<RT, FT> {
    label?: string | undefined;
    default?: FT | undefined;
    value: FieldValueGetter<RT, FT>;
}
export declare enum FormatterTypes {
    header = "header",
    undefined = "undefined",
    boolean = "boolean",
    number = "number",
    bigint = "bigint",
    string = "string",
    symbol = "symbol",
    function = "function",
    object = "object"
}
export interface FormattersOptions {
    [FormatterTypes.header]?: Formatter<string>;
    [FormatterTypes.undefined]?: Formatter<undefined>;
    [FormatterTypes.boolean]?: Formatter<boolean>;
    [FormatterTypes.number]?: Formatter<number>;
    [FormatterTypes.bigint]?: Formatter<bigint>;
    [FormatterTypes.string]?: Formatter<string>;
    [FormatterTypes.symbol]?: Formatter<symbol>;
    [FormatterTypes.function]?: Formatter<Function>;
    [FormatterTypes.object]?: Formatter<object>;
}
export interface Json2CSVBaseOptions<TRaw, T> {
    fields?: Array<string | FieldInfo<T, unknown>>;
    ndjson?: boolean;
    defaultValue?: string;
    delimiter?: string;
    eol?: string;
    header?: boolean;
    includeEmptyRows?: boolean;
    withBOM?: boolean;
    formatters?: FormattersOptions;
    transforms?: [] | [Transform<TRaw, T>] | [Transform<TRaw, any>, ...Array<Transform<any, any>>, Transform<any, T>];
}
interface NormalizedFieldInfo<RT, FT> {
    label: string;
    value: FieldValueGetterFnWithoutField<RT, FT>;
}
export interface NormalizedJson2CSVBaseOptions<TRaw, T> extends Required<Json2CSVBaseOptions<TRaw, T>> {
    fields: Array<NormalizedFieldInfo<T, unknown>>;
    formatters: Required<FormattersOptions>;
}
export default abstract class JSON2CSVBase<TRaw extends object, T extends object> {
    protected opts: NormalizedJson2CSVBaseOptions<TRaw, T>;
    constructor(opts?: Readonly<Json2CSVBaseOptions<TRaw, T>>);
    /**
     * Check passing opts and set defaults.
     *
     * @param {Json2CsvOptions} opts Options object containing fields,
     * delimiter, default value, header, etc.
     */
    protected preprocessOpts(opts?: Json2CSVBaseOptions<TRaw, T>): NormalizedJson2CSVBaseOptions<TRaw, T>;
    /**
     * Check and normalize the fields configuration.
     *
     * @param {(string|object)[]} fields Fields configuration provided by the user
     * or inferred from the data
     * @returns {object[]} preprocessed FieldsInfo array
     */
    protected preprocessFieldsInfo(fields: Array<string | FieldInfo<T, unknown>>, globalDefaultValue?: string): Array<NormalizedFieldInfo<T, unknown>>;
    /**
     * Create the title row with all the provided fields as column headings
     *
     * @returns {String} titles as a string
     */
    protected getHeader(): string;
    /**
     * Preprocess each object according to the given transforms (unwind, flatten, etc.).
     * @param {Object} row JSON object to be converted in a CSV row
     */
    protected preprocessRow(row: TRaw): Array<T>;
    /**
     * Create the content of a specific CSV row
     *
     * @param {Object} row JSON object to be converted in a CSV row
     * @returns {String} CSV string (row)
     */
    protected processRow(row: T): string | undefined;
    /**
     * Create the content of a specfic CSV row cell
     *
     * @param {Object} row JSON object representing the  CSV row that the cell belongs to
     * @param {FieldInfo} fieldInfo Details of the field to process to be a CSV cell
     * @returns {String} CSV string (cell)
     */
    protected processCell<FT>(row: T, fieldInfo: NormalizedFieldInfo<T, FT>): string;
    /**
     * Create the content of a specfic CSV row cell
     *
     * @param {T} value Value to be included in a CSV cell
     * @returns {String} Value stringified and processed
     */
    protected processValue<T>(value: T): string;
}
export {};
//# sourceMappingURL=BaseParser.d.ts.map