const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Property = sequelize.define('Property', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  title: {
    type: DataTypes.STRING(255),
    allowNull: false
  },
  description: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  type: {
    type: DataTypes.ENUM('apartment', 'house', 'condo', 'townhouse', 'studio', 'villa'),
    allowNull: false
  },
  price: {
    type: DataTypes.DECIMAL(10, 2),
    allowNull: false
  },
  bedrooms: {
    type: DataTypes.INTEGER,
    allowNull: false,
    defaultValue: 0
  },
  bathrooms: {
    type: DataTypes.DECIMAL(3, 1),
    allowNull: false,
    defaultValue: 0
  },
  area: {
    type: DataTypes.DECIMAL(8, 2),
    allowNull: true,
    comment: 'Area in square feet'
  },
  address: {
    type: DataTypes.STRING(500),
    allowNull: false
  },
  city: {
    type: DataTypes.STRING(100),
    allowNull: false
  },
  state: {
    type: DataTypes.STRING(100),
    allowNull: false
  },
  zipCode: {
    type: DataTypes.STRING(20),
    allowNull: false,
    field: 'zip_code'
  },
  country: {
    type: DataTypes.STRING(100),
    allowNull: false,
    defaultValue: 'USA'
  },
  latitude: {
    type: DataTypes.DECIMAL(10, 8),
    allowNull: true
  },
  longitude: {
    type: DataTypes.DECIMAL(11, 8),
    allowNull: true
  },
  images: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: []
  },
  amenities: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: []
  },
  status: {
    type: DataTypes.ENUM('available', 'occupied', 'maintenance', 'unavailable'),
    allowNull: false,
    defaultValue: 'available'
  },
  isPublished: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: false,
    field: 'is_published'
  },
  landlordId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'landlord_id',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  features: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: {}
  },
  rules: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: {}
  },
  depositAmount: {
    type: DataTypes.DECIMAL(10, 2),
    allowNull: true,
    field: 'deposit_amount'
  },
  availableFrom: {
    type: DataTypes.DATE,
    allowNull: true,
    field: 'available_from'
  }
}, {
  tableName: 'properties',
  timestamps: true
});

// Instance methods
Property.prototype.getFullAddress = function() {
  return `${this.address}, ${this.city}, ${this.state} ${this.zipCode}`;
};

Property.prototype.isAvailable = function() {
  return this.status === 'available' && this.isPublished;
};

// Static methods
Property.getAvailableProperties = function() {
  return this.findAll({
    where: {
      status: 'available',
      isPublished: true
    }
  });
};

Property.findByLocation = function(city, state) {
  return this.findAll({
    where: {
      city: city.toLowerCase(),
      state: state.toUpperCase(),
      status: 'available',
      isPublished: true
    }
  });
};

module.exports = Property;