const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Payment = sequelize.define('Payment', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  amount: {
    type: DataTypes.DECIMAL(10, 2),
    allowNull: false
  },
  dueDate: {
    type: DataTypes.DATE,
    allowNull: false,
    field: 'due_date'
  },
  paidDate: {
    type: DataTypes.DATE,
    allowNull: true,
    field: 'paid_date'
  },
  status: {
    type: DataTypes.ENUM('pending', 'paid', 'overdue', 'failed', 'refunded'),
    allowNull: false,
    defaultValue: 'pending'
  },
  paymentMethod: {
    type: DataTypes.ENUM('credit_card', 'debit_card', 'bank_transfer', 'cash', 'check', 'online'),
    allowNull: true,
    field: 'payment_method'
  },
  transactionId: {
    type: DataTypes.STRING(255),
    allowNull: true,
    field: 'transaction_id'
  },
  receiptUrl: {
    type: DataTypes.STRING(500),
    allowNull: true,
    field: 'receipt_url'
  },
  description: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  lateFee: {
    type: DataTypes.DECIMAL(10, 2),
    allowNull: true,
    defaultValue: 0,
    field: 'late_fee'
  },
  leaseId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'lease_id',
    references: {
      model: 'leases',
      key: 'id'
    }
  },
  tenantId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'tenant_id',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  landlordId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'landlord_id',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  propertyId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'property_id',
    references: {
      model: 'properties',
      key: 'id'
    }
  }
}, {
  tableName: 'payments',
  timestamps: true
});

// Instance methods
Payment.prototype.isOverdue = function() {
  const now = new Date();
  return this.status === 'pending' && this.dueDate < now;
};

Payment.prototype.calculateLateFee = function() {
  if (!this.isOverdue()) return 0;
  
  const now = new Date();
  const dueDate = new Date(this.dueDate);
  const daysLate = Math.ceil((now - dueDate) / (1000 * 60 * 60 * 24));
  
  // Example: $50 late fee plus $10 per day after 5 days
  if (daysLate <= 5) return 50;
  return 50 + (daysLate - 5) * 10;
};

// Static methods
Payment.getOverduePayments = function() {
  const now = new Date();
  return this.findAll({
    where: {
      status: 'pending',
      dueDate: { [Op.lt]: now }
    }
  });
};

Payment.getMonthlyRevenue = function(landlordId, year, month) {
  const startDate = new Date(year, month - 1, 1);
  const endDate = new Date(year, month, 0);
  
  return this.sum('amount', {
    where: {
      landlordId,
      status: 'paid',
      paidDate: {
        [Op.between]: [startDate, endDate]
      }
    }
  });
};

module.exports = Payment;