const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Notification = sequelize.define('Notification', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  title: {
    type: DataTypes.STRING(255),
    allowNull: false
  },
  message: {
    type: DataTypes.TEXT,
    allowNull: false
  },
  type: {
    type: DataTypes.ENUM(
      'payment_reminder',
      'maintenance_update', 
      'lease_expiry',
      'new_message',
      'system_alert',
      'booking_confirmation',
      'payment_received',
      'maintenance_scheduled'
    ),
    allowNull: false
  },
  isRead: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: false,
    field: 'is_read'
  },
  priority: {
    type: DataTypes.ENUM('low', 'medium', 'high'),
    allowNull: false,
    defaultValue: 'medium'
  },
  metadata: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: {}
  },
  userId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'user_id',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  relatedEntityType: {
    type: DataTypes.ENUM('property', 'lease', 'payment', 'maintenance', 'user'),
    allowNull: true,
    field: 'related_entity_type'
  },
  relatedEntityId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'related_entity_id'
  },
  expiresAt: {
    type: DataTypes.DATE,
    allowNull: true,
    field: 'expires_at'
  }
}, {
  tableName: 'notifications',
  timestamps: true,
  indexes: [
    {
      fields: ['user_id', 'is_read']
    },
    {
      fields: ['created_at']
    }
  ]
});

// Instance methods
Notification.prototype.markAsRead = function() {
  this.isRead = true;
  return this.save();
};

Notification.prototype.isExpired = function() {
  if (!this.expiresAt) return false;
  return new Date() > new Date(this.expiresAt);
};

// Static methods
Notification.getUnreadCount = function(userId) {
  return this.count({
    where: {
      userId,
      isRead: false
    }
  });
};

Notification.getUserNotifications = function(userId, limit = 20) {
  return this.findAll({
    where: { userId },
    order: [['createdAt', 'DESC']],
    limit
  });
};

Notification.createSystemNotification = function(userId, title, message, type = 'system_alert', metadata = {}) {
  return this.create({
    userId,
    title,
    message,
    type,
    metadata,
    priority: 'medium'
  });
};

module.exports = Notification;