const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Maintenance = sequelize.define('Maintenance', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  title: {
    type: DataTypes.STRING(255),
    allowNull: false
  },
  description: {
    type: DataTypes.TEXT,
    allowNull: false
  },
  priority: {
    type: DataTypes.ENUM('low', 'medium', 'high', 'emergency'),
    allowNull: false,
    defaultValue: 'medium'
  },
  status: {
    type: DataTypes.ENUM('pending', 'in_progress', 'completed', 'cancelled'),
    allowNull: false,
    defaultValue: 'pending'
  },
  category: {
    type: DataTypes.ENUM('plumbing', 'electrical', 'hvac', 'appliance', 'structural', 'cosmetic', 'pest_control', 'other'),
    allowNull: false,
    defaultValue: 'other'
  },
  images: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: []
  },
  scheduledDate: {
    type: DataTypes.DATE,
    allowNull: true,
    field: 'scheduled_date'
  },
  completedDate: {
    type: DataTypes.DATE,
    allowNull: true,
    field: 'completed_date'
  },
  cost: {
    type: DataTypes.DECIMAL(10, 2),
    allowNull: true
  },
  notes: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  tenantId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'tenant_id',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  propertyId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'property_id',
    references: {
      model: 'properties',
      key: 'id'
    }
  },
  assignedTo: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'assigned_to',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  landlordId: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'landlord_id',
    references: {
      model: 'users',
      key: 'id'
    }
  }
}, {
  tableName: 'maintenance_requests',
  timestamps: true
});

// Instance methods
Maintenance.prototype.isOverdue = function() {
  if (this.scheduledDate && this.status === 'pending') {
    return new Date() > new Date(this.scheduledDate);
  }
  return false;
};

Maintenance.prototype.canBeCompleted = function() {
  return this.status === 'in_progress' || this.status === 'pending';
};

// Static methods
Maintenance.getHighPriorityRequests = function() {
  return this.findAll({
    where: {
      priority: ['high', 'emergency'],
      status: ['pending', 'in_progress']
    },
    order: [['createdAt', 'ASC']]
  });
};

Maintenance.getRequestsByStatus = function(status) {
  return this.findAll({
    where: { status },
    include: ['property', 'tenant', 'assignedStaff']
  });
};

module.exports = Maintenance;