const { DataTypes } = require('sequelize');
const { sequelize } = require('../config/database');

const Document = sequelize.define('Document', {
  id: {
    type: DataTypes.INTEGER,
    primaryKey: true,
    autoIncrement: true
  },
  title: {
    type: DataTypes.STRING(255),
    allowNull: false
  },
  description: {
    type: DataTypes.TEXT,
    allowNull: true
  },
  fileUrl: {
    type: DataTypes.STRING(500),
    allowNull: false,
    field: 'file_url'
  },
  fileName: {
    type: DataTypes.STRING(255),
    allowNull: false,
    field: 'file_name'
  },
  fileSize: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'file_size'
  },
  mimeType: {
    type: DataTypes.STRING(100),
    allowNull: true,
    field: 'mime_type'
  },
  documentType: {
    type: DataTypes.ENUM(
      'lease_agreement',
      'id_proof',
      'income_proof',
      'property_document',
      'maintenance_report',
      'inspection_report',
      'insurance_document',
      'other'
    ),
    allowNull: false,
    field: 'document_type'
  },
  category: {
    type: DataTypes.STRING(100),
    allowNull: true
  },
  tags: {
    type: DataTypes.JSON,
    allowNull: true,
    defaultValue: []
  },
  isPublic: {
    type: DataTypes.BOOLEAN,
    allowNull: false,
    defaultValue: false,
    field: 'is_public'
  },
  expiresAt: {
    type: DataTypes.DATE,
    allowNull: true,
    field: 'expires_at'
  },
  uploadedBy: {
    type: DataTypes.INTEGER,
    allowNull: false,
    field: 'uploaded_by',
    references: {
      model: 'users',
      key: 'id'
    }
  },
  propertyId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'property_id',
    references: {
      model: 'properties',
      key: 'id'
    }
  },
  leaseId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'lease_id',
    references: {
      model: 'leases',
      key: 'id'
    }
  },
  userId: {
    type: DataTypes.INTEGER,
    allowNull: true,
    field: 'user_id',
    references: {
      model: 'users',
      key: 'id'
    }
  }
}, {
  tableName: 'documents',
  timestamps: true,
  indexes: [
    {
      fields: ['document_type']
    },
    {
      fields: ['property_id']
    },
    {
      fields: ['lease_id']
    }
  ]
});

// Instance methods
Document.prototype.isExpired = function() {
  if (!this.expiresAt) return false;
  return new Date() > new Date(this.expiresAt);
};

Document.prototype.getFileExtension = function() {
  return this.fileName.split('.').pop().toLowerCase();
};

// Static methods
Document.getLeaseDocuments = function(leaseId) {
  return this.findAll({
    where: { leaseId },
    order: [['createdAt', 'DESC']]
  });
};

Document.getPropertyDocuments = function(propertyId, documentType = null) {
  const where = { propertyId };
  if (documentType) where.documentType = documentType;
  
  return this.findAll({
    where,
    order: [['createdAt', 'DESC']]
  });
};

Document.getUserDocuments = function(userId) {
  return this.findAll({
    where: { userId },
    order: [['createdAt', 'DESC']]
  });
};

module.exports = Document;