const multer = require('multer');
const path = require('path');

// Configure multer for memory storage
const storage = multer.memoryStorage();

const fileFilter = (req, file, cb) => {
  // Check file type
  const allowedTypes = {
    'image/jpeg': true,
    'image/jpg': true,
    'image/png': true,
    'image/webp': true,
    'application/pdf': true,
    'application/msword': true,
    'application/vnd.openxmlformats-officedocument.wordprocessingml.document': true
  };

  if (allowedTypes[file.mimetype]) {
    cb(null, true);
  } else {
    cb(new Error('Invalid file type. Only images and documents are allowed.'), false);
  }
};

const upload = multer({
  storage: storage,
  fileFilter: fileFilter,
  limits: {
    fileSize: 10 * 1024 * 1024, // 10MB limit
    files: 10 // Maximum 10 files
  }
});

const handleUpload = (fieldName = 'files', maxCount = 10) => {
  return upload.array(fieldName, maxCount);
};

// Simple upload middleware for now (without Cloudinary)
const uploadToCloudinaryMiddleware = async (req, res, next) => {
  try {
    if (!req.files || req.files.length === 0) {
      return next();
    }

    // For now, just pass the files through
    // In production, you'd upload to Cloudinary here
    req.uploadedFiles = req.files.map(file => ({
      url: `uploads/${file.originalname}`, // Placeholder URL
      filename: file.originalname,
      mimetype: file.mimetype,
      size: file.size
    }));

    next();
  } catch (error) {
    next(error);
  }
};

const setUploadFolder = (folder) => {
  return (req, res, next) => {
    req.uploadFolder = folder;
    next();
  };
};

module.exports = {
  handleUpload,
  uploadToCloudinaryMiddleware,
  setUploadFolder
};