const xss = require('xss');
const sanitizeHtml = require('sanitize-html');

const sanitizeOptions = {
  allowedTags: [
    'h1', 'h2', 'h3', 'h4', 'h5', 'h6',
    'blockquote', 'p', 'a', 'ul', 'ol', 'nl', 'li',
    'b', 'i', 'strong', 'em', 'strike', 'code', 'hr', 'br', 'div',
    'table', 'thead', 'caption', 'tbody', 'tr', 'th', 'td', 'pre',
    'span'
  ],
  allowedAttributes: {
    a: ['href', 'name', 'target'],
    img: ['src', 'alt'],
    '*': ['class', 'id', 'style']
  },
  allowedStyles: {
    '*': {
      'color': [/^#(0x)?[0-9a-f]+$/i, /^rgb\(\s*(\d{1,3})\s*,\s*(\d{1,3})\s*,\s*(\d{1,3})\s*\)$/],
      'text-align': [/^left$/, /^right$/, /^center$/],
      'font-size': [/^\d+(?:px|em|%)$/]
    }
  },
  selfClosing: ['img', 'br', 'hr', 'area', 'base', 'basefont', 'input', 'link', 'meta'],
  allowedSchemes: ['http', 'https', 'mailto'],
  allowedSchemesByTag: {},
  allowProtocolRelative: true
};

const sanitize = (obj, options = sanitizeOptions) => {
  if (typeof obj === 'string') {
    return sanitizeHtml(obj, options);
  } else if (Array.isArray(obj)) {
    return obj.map(item => sanitize(item, options));
  } else if (typeof obj === 'object' && obj !== null) {
    const sanitized = {};
    for (const key in obj) {
      sanitized[key] = sanitize(obj[key], options);
    }
    return sanitized;
  }
  return obj;
};

const sanitizeMiddleware = (req, res, next) => {
  if (req.body) {
    req.body = sanitize(req.body);
  }
  if (req.query) {
    req.query = sanitize(req.query);
  }
  if (req.params) {
    req.params = sanitize(req.params);
  }
  next();
};

const sanitizeString = (input) => {
  if (typeof input !== 'string') return input;
  
  // Remove potentially dangerous characters
  return input
    .replace(/[<>]/g, '') // Remove < and >
    .replace(/javascript:/gi, '') // Remove javascript: protocol
    .replace(/on\w+=/gi, '') // Remove event handlers
    .trim();
};

const sanitizeEmail = (email) => {
  if (typeof email !== 'string') return email;
  
  return email
    .toLowerCase()
    .trim()
    .replace(/[^a-zA-Z0-9@._+-]/g, '');
};

const sanitizePhone = (phone) => {
  if (typeof phone !== 'string') return phone;
  
  return phone.replace(/[^\d+()-]/g, '');
};

const sanitizeNumber = (input) => {
  if (typeof input === 'number') return input;
  if (typeof input !== 'string') return null;
  
  const cleaned = input.replace(/[^\d.-]/g, '');
  const num = parseFloat(cleaned);
  return isNaN(num) ? null : num;
};

const sanitizeUrl = (url) => {
  if (typeof url !== 'string') return url;
  
  try {
    const parsed = new URL(url);
    const allowedProtocols = ['http:', 'https:', 'mailto:'];
    
    if (!allowedProtocols.includes(parsed.protocol)) {
      return null;
    }
    
    return parsed.toString();
  } catch {
    return null;
  }
};

const sanitizeObject = (obj, schema) => {
  const sanitized = {};
  
  for (const [key, config] of Object.entries(schema)) {
    if (obj.hasOwnProperty(key)) {
      const value = obj[key];
      
      switch (config.type) {
        case 'string':
          sanitized[key] = sanitizeString(value);
          break;
        case 'email':
          sanitized[key] = sanitizeEmail(value);
          break;
        case 'phone':
          sanitized[key] = sanitizePhone(value);
          break;
        case 'number':
          sanitized[key] = sanitizeNumber(value);
          break;
        case 'url':
          sanitized[key] = sanitizeUrl(value);
          break;
        case 'html':
          sanitized[key] = sanitizeHtml(value, sanitizeOptions);
          break;
        default:
          sanitized[key] = value;
      }
      
      // Apply length limits
      if (config.maxLength && typeof sanitized[key] === 'string') {
        sanitized[key] = sanitized[key].substring(0, config.maxLength);
      }
    }
  }
  
  return sanitized;
};

// Common sanitization schemas
const userSanitizationSchema = {
  firstName: { type: 'string', maxLength: 100 },
  lastName: { type: 'string', maxLength: 100 },
  email: { type: 'email' },
  phone: { type: 'phone' },
  address: { type: 'string' }
};

const propertySanitizationSchema = {
  name: { type: 'string', maxLength: 255 },
  description: { type: 'html' },
  address: { type: 'string' }
};

const sanitizeUserInput = (req, res, next) => {
  if (req.body) {
    req.body = sanitizeObject(req.body, userSanitizationSchema);
  }
  next();
};

const sanitizePropertyInput = (req, res, next) => {
  if (req.body) {
    req.body = sanitizeObject(req.body, propertySanitizationSchema);
  }
  next();
};

module.exports = {
  sanitize,
  sanitizeMiddleware,
  sanitizeString,
  sanitizeEmail,
  sanitizePhone,
  sanitizeNumber,
  sanitizeUrl,
  sanitizeObject,
  sanitizeUserInput,
  sanitizePropertyInput,
  sanitizeOptions
};