const { deleteFromCloudinary } = require('../config/cloudinary');

const uploadController = {
  // Upload single file
  uploadSingle: async (req, res) => {
    try {
      if (!req.uploadedFiles || req.uploadedFiles.length === 0) {
        return res.status(400).json({
          success: false,
          message: 'No file uploaded'
        });
      }

      const file = req.uploadedFiles[0];

      res.json({
        success: true,
        message: 'File uploaded successfully',
        data: {
          file: {
            url: file.url,
            public_id: file.public_id,
            format: file.format,
            bytes: file.bytes,
            width: file.width,
            height: file.height
          }
        }
      });
    } catch (error) {
      console.error('Upload single file error:', error);
      res.status(500).json({
        success: false,
        message: 'Error uploading file'
      });
    }
  },

  // Upload multiple files
  uploadMultiple: async (req, res) => {
    try {
      if (!req.uploadedFiles || req.uploadedFiles.length === 0) {
        return res.status(400).json({
          success: false,
          message: 'No files uploaded'
        });
      }

      res.json({
        success: true,
        message: 'Files uploaded successfully',
        data: {
          files: req.uploadedFiles.map(file => ({
            url: file.url,
            public_id: file.public_id,
            format: file.format,
            bytes: file.bytes,
            width: file.width,
            height: file.height
          }))
        }
      });
    } catch (error) {
      console.error('Upload multiple files error:', error);
      res.status(500).json({
        success: false,
        message: 'Error uploading files'
      });
    }
  },

  // Upload property images
  uploadPropertyImages: async (req, res) => {
    try {
      if (!req.uploadedFiles || req.uploadedFiles.length === 0) {
        return res.status(400).json({
          success: false,
          message: 'No images uploaded'
        });
      }

      res.json({
        success: true,
        message: 'Property images uploaded successfully',
        data: {
          images: req.uploadedFiles.map(file => ({
            url: file.url,
            public_id: file.public_id,
            format: file.format,
            bytes: file.bytes,
            width: file.width,
            height: file.height
          }))
        }
      });
    } catch (error) {
      console.error('Upload property images error:', error);
      res.status(500).json({
        success: false,
        message: 'Error uploading property images'
      });
    }
  },

  // Upload document
  uploadDocument: async (req, res) => {
    try {
      if (!req.uploadedFiles || req.uploadedFiles.length === 0) {
        return res.status(400).json({
          success: false,
          message: 'No document uploaded'
        });
      }

      const document = req.uploadedFiles[0];

      res.json({
        success: true,
        message: 'Document uploaded successfully',
        data: {
          document: {
            url: document.url,
            public_id: document.public_id,
            format: document.format,
            bytes: document.bytes,
            fileName: req.files[0].originalname
          }
        }
      });
    } catch (error) {
      console.error('Upload document error:', error);
      res.status(500).json({
        success: false,
        message: 'Error uploading document'
      });
    }
  },

  // Upload avatar
  uploadAvatar: async (req, res) => {
    try {
      if (!req.uploadedFiles || req.uploadedFiles.length === 0) {
        return res.status(400).json({
          success: false,
          message: 'No avatar image uploaded'
        });
      }

      const avatar = req.uploadedFiles[0];

      res.json({
        success: true,
        message: 'Avatar uploaded successfully',
        data: {
          avatar: {
            url: avatar.url,
            public_id: avatar.public_id,
            format: avatar.format,
            bytes: avatar.bytes,
            width: avatar.width,
            height: avatar.height
          }
        }
      });
    } catch (error) {
      console.error('Upload avatar error:', error);
      res.status(500).json({
        success: false,
        message: 'Error uploading avatar'
      });
    }
  },

  // Delete uploaded file
  deleteFile: async (req, res) => {
    try {
      const { publicId } = req.params;

      if (!publicId) {
        return res.status(400).json({
          success: false,
          message: 'Public ID is required'
        });
      }

      const result = await deleteFromCloudinary(publicId);

      if (result.result === 'ok') {
        res.json({
          success: true,
          message: 'File deleted successfully'
        });
      } else {
        res.status(400).json({
          success: false,
          message: 'Failed to delete file'
        });
      }
    } catch (error) {
      console.error('Delete file error:', error);
      res.status(500).json({
        success: false,
        message: 'Error deleting file'
      });
    }
  },

  // Get upload configuration
  getUploadConfig: async (req, res) => {
    try {
      const config = {
        maxFileSize: 10 * 1024 * 1024, // 10MB
        maxFiles: 10,
        allowedTypes: [
          'image/jpeg',
          'image/jpg',
          'image/png',
          'image/webp',
          'application/pdf',
          'application/msword',
          'application/vnd.openxmlformats-officedocument.wordprocessingml.document'
        ],
        uploadFolders: {
          properties: 'stayspot/properties',
          documents: 'stayspot/documents',
          avatars: 'stayspot/avatars',
          maintenance: 'stayspot/maintenance'
        }
      };

      res.json({
        success: true,
        data: { config }
      });
    } catch (error) {
      console.error('Get upload config error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching upload configuration'
      });
    }
  }
};

module.exports = uploadController;