const { Payment, Lease, Property, User, Maintenance } = require('../models');
const { Op } = require('sequelize');
const moment = require('moment');
const CSVExporter = require('../utils/csvExporter');

const reportController = {
  // Generate financial report
  generateFinancialReport: async (req, res) => {
    try {
      const { startDate, endDate, landlordId, format = 'json' } = req.query;
      const where = { status: 'paid' };

      if (landlordId) {
        where.landlordId = landlordId;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      if (startDate && endDate) {
        where.paidDate = {
          [Op.between]: [new Date(startDate), new Date(endDate)]
        };
      }

      const payments = await Payment.findAll({
        where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email']
          },
          {
            model: Lease,
            as: 'lease',
            attributes: ['id', 'startDate', 'endDate']
          }
        ],
        order: [['paidDate', 'DESC']]
      });

      const totalRevenue = payments.reduce((sum, payment) => sum + parseFloat(payment.amount), 0);

      const report = {
        summary: {
          totalRevenue,
          totalPayments: payments.length,
          period: {
            startDate: startDate || 'Beginning',
            endDate: endDate || 'Present'
          },
          averagePayment: payments.length > 0 ? totalRevenue / payments.length : 0
        },
        payments
      };

      if (format === 'csv') {
        const csv = CSVExporter.generatePaymentsCSV(payments);
        res.setHeader('Content-Type', 'text/csv');
        res.setHeader('Content-Disposition', `attachment; filename=financial-report-${Date.now()}.csv`);
        return res.send(csv);
      }

      res.json({
        success: true,
        data: report
      });
    } catch (error) {
      console.error('Generate financial report error:', error);
      res.status(500).json({
        success: false,
        message: 'Error generating financial report'
      });
    }
  },

  // Generate occupancy report
  generateOccupancyReport: async (req, res) => {
    try {
      const { landlordId } = req.query;
      const where = {};

      if (landlordId) {
        where.landlordId = landlordId;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      const properties = await Property.findAll({
        where,
        include: [
          {
            model: Lease,
            as: 'leases',
            where: { status: 'active' },
            required: false,
            include: [{
              model: User,
              as: 'tenant',
              attributes: ['id', 'firstName', 'lastName', 'email']
            }]
          }
        ]
      });

      const totalProperties = properties.length;
      const occupiedProperties = properties.filter(property => property.leases.length > 0).length;
      const vacancyRate = ((totalProperties - occupiedProperties) / totalProperties) * 100;

      const report = {
        summary: {
          totalProperties,
          occupiedProperties,
          vacantProperties: totalProperties - occupiedProperties,
          occupancyRate: ((occupiedProperties / totalProperties) * 100).toFixed(2),
          vacancyRate: vacancyRate.toFixed(2)
        },
        properties: properties.map(property => ({
          id: property.id,
          title: property.title,
          address: property.address,
          status: property.status,
          isOccupied: property.leases.length > 0,
          currentLease: property.leases[0] || null
        }))
      };

      res.json({
        success: true,
        data: report
      });
    } catch (error) {
      console.error('Generate occupancy report error:', error);
      res.status(500).json({
        success: false,
        message: 'Error generating occupancy report'
      });
    }
  },

  // Generate maintenance report
  generateMaintenanceReport: async (req, res) => {
    try {
      const { startDate, endDate, landlordId, status } = req.query;
      const where = {};

      if (landlordId) {
        where.landlordId = landlordId;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      if (status) where.status = status;

      if (startDate && endDate) {
        where.createdAt = {
          [Op.between]: [new Date(startDate), new Date(endDate)]
        };
      }

      const maintenanceRequests = await Maintenance.findAll({
        where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email']
          },
          {
            model: User,
            as: 'assignedStaff',
            attributes: ['id', 'firstName', 'lastName', 'email']
          }
        ],
        order: [['createdAt', 'DESC']]
      });

      const stats = {
        total: maintenanceRequests.length,
        pending: maintenanceRequests.filter(req => req.status === 'pending').length,
        inProgress: maintenanceRequests.filter(req => req.status === 'in_progress').length,
        completed: maintenanceRequests.filter(req => req.status === 'completed').length,
        cancelled: maintenanceRequests.filter(req => req.status === 'cancelled').length
      };

      const report = {
        summary: stats,
        maintenanceRequests
      };

      res.json({
        success: true,
        data: report
      });
    } catch (error) {
      console.error('Generate maintenance report error:', error);
      res.status(500).json({
        success: false,
        message: 'Error generating maintenance report'
      });
    }
  },

  // Generate tenant report
  generateTenantReport: async (req, res) => {
    try {
      const { landlordId, activeOnly = true } = req.query;
      const where = {};

      if (landlordId) {
        where.landlordId = landlordId;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      const leases = await Lease.findAll({
        where: activeOnly ? { ...where, status: 'active' } : where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address', 'price']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email', 'phone', 'createdAt']
          }
        ],
        order: [['startDate', 'DESC']]
      });

      const tenants = leases.map(lease => ({
        id: lease.tenant.id,
        firstName: lease.tenant.firstName,
        lastName: lease.tenant.lastName,
        email: lease.tenant.email,
        phone: lease.tenant.phone,
        leaseStart: lease.startDate,
        leaseEnd: lease.endDate,
        property: lease.property.title,
        address: lease.property.address,
        monthlyRent: lease.monthlyRent
      }));

      const report = {
        summary: {
          totalTenants: tenants.length,
          totalMonthlyRent: tenants.reduce((sum, tenant) => sum + parseFloat(tenant.monthlyRent), 0)
        },
        tenants
      };

      res.json({
        success: true,
        data: report
      });
    } catch (error) {
      console.error('Generate tenant report error:', error);
      res.status(500).json({
        success: false,
        message: 'Error generating tenant report'
      });
    }
  },

  // Generate property performance report
  generatePropertyPerformanceReport: async (req, res) => {
    try {
      const { landlordId } = req.query;
      const where = {};

      if (landlordId) {
        where.landlordId = landlordId;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      const properties = await Property.findAll({
        where,
        include: [
          {
            model: Lease,
            as: 'leases',
            where: { status: 'active' },
            required: false
          },
          {
            model: Payment,
            as: 'payments',
            where: { status: 'paid' },
            required: false
          },
          {
            model: Maintenance,
            as: 'maintenances',
            required: false
          }
        ]
      });

      const performanceData = properties.map(property => {
        const totalRevenue = property.payments.reduce((sum, payment) => sum + parseFloat(payment.amount), 0);
        const maintenanceCount = property.maintenances.length;
        const occupancyRate = property.leases.length > 0 ? 100 : 0;

        return {
          id: property.id,
          title: property.title,
          address: property.address,
          price: property.price,
          status: property.status,
          occupancyRate,
          totalRevenue,
          maintenanceCount,
          isOccupied: property.leases.length > 0
        };
      });

      const report = {
        summary: {
          totalProperties: properties.length,
          occupiedProperties: properties.filter(p => p.leases.length > 0).length,
          totalRevenue: performanceData.reduce((sum, property) => sum + property.totalRevenue, 0),
          averageOccupancyRate: (performanceData.reduce((sum, property) => sum + property.occupancyRate, 0) / performanceData.length).toFixed(2)
        },
        properties: performanceData
      };

      res.json({
        success: true,
        data: report
      });
    } catch (error) {
      console.error('Generate property performance report error:', error);
      res.status(500).json({
        success: false,
        message: 'Error generating property performance report'
      });
    }
  },

  // Generate lease expiry report
  generateLeaseExpiryReport: async (req, res) => {
    try {
      const { days = 30, landlordId } = req.query;
      const where = {
        status: 'active',
        endDate: {
          [Op.between]: [new Date(), new Date(Date.now() + days * 24 * 60 * 60 * 1000)]
        }
      };

      if (landlordId) {
        where.landlordId = landlordId;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      const expiringLeases = await Lease.findAll({
        where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
          }
        ],
        order: [['endDate', 'ASC']]
      });

      const report = {
        summary: {
          totalExpiring: expiringLeases.length,
          daysThreshold: parseInt(days)
        },
        expiringLeases: expiringLeases.map(lease => ({
          id: lease.id,
          endDate: lease.endDate,
          daysRemaining: Math.ceil((new Date(lease.endDate) - new Date()) / (1000 * 60 * 60 * 24)),
          property: lease.property.title,
          address: lease.property.address,
          tenant: `${lease.tenant.firstName} ${lease.tenant.lastName}`,
          email: lease.tenant.email,
          phone: lease.tenant.phone
        }))
      };

      res.json({
        success: true,
        data: report
      });
    } catch (error) {
      console.error('Generate lease expiry report error:', error);
      res.status(500).json({
        success: false,
        message: 'Error generating lease expiry report'
      });
    }
  },

  // Generate payment overdue report
  generatePaymentOverdueReport: async (req, res) => {
    try {
      const { landlordId } = req.query;
      const where = {
        status: 'pending',
        dueDate: { [Op.lt]: new Date() }
      };

      if (landlordId) {
        where.landlordId = landlordId;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      const overduePayments = await Payment.findAll({
        where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
          },
          {
            model: Lease,
            as: 'lease',
            attributes: ['id', 'startDate', 'endDate']
          }
        ],
        order: [['dueDate', 'ASC']]
      });

      const totalOverdue = overduePayments.reduce((sum, payment) => sum + parseFloat(payment.amount), 0);

      const report = {
        summary: {
          totalOverduePayments: overduePayments.length,
          totalOverdueAmount: totalOverdue
        },
        overduePayments: overduePayments.map(payment => ({
          id: payment.id,
          amount: payment.amount,
          dueDate: payment.dueDate,
          daysOverdue: Math.ceil((new Date() - new Date(payment.dueDate)) / (1000 * 60 * 60 * 24)),
          property: payment.property.title,
          address: payment.property.address,
          tenant: `${payment.tenant.firstName} ${payment.tenant.lastName}`,
          email: payment.tenant.email,
          phone: payment.tenant.phone
        }))
      };

      res.json({
        success: true,
        data: report
      });
    } catch (error) {
      console.error('Generate payment overdue report error:', error);
      res.status(500).json({
        success: false,
        message: 'Error generating payment overdue report'
      });
    }
  },

  // Generate custom report
  generateCustomReport: async (req, res) => {
    try {
      const { reportType, filters, fields } = req.body;

      // This is a simplified custom report generator
      // In a real application, this would be more sophisticated
      let reportData = {};

      switch (reportType) {
        case 'financial_summary':
          reportData = await generateFinancialSummary(filters);
          break;
        case 'property_performance':
          reportData = await generatePropertyPerformance(filters);
          break;
        case 'tenant_activity':
          reportData = await generateTenantActivity(filters);
          break;
        default:
          return res.status(400).json({
            success: false,
            message: 'Unsupported report type'
          });
      }

      res.json({
        success: true,
        data: reportData
      });
    } catch (error) {
      console.error('Generate custom report error:', error);
      res.status(500).json({
        success: false,
        message: 'Error generating custom report'
      });
    }
  },

  // Export report data
  exportReportData: async (req, res) => {
    try {
      const { format } = req.params;
      const { reportType, filters } = req.query;

      if (format !== 'csv' && format !== 'json') {
        return res.status(400).json({
          success: false,
          message: 'Unsupported export format. Use csv or json.'
        });
      }

      // This would generate and export the actual data
      // For now, return a placeholder response
      res.json({
        success: true,
        message: `Export functionality for ${format} format would be implemented here`,
        data: {
          format,
          reportType,
          filters
        }
      });
    } catch (error) {
      console.error('Export report data error:', error);
      res.status(500).json({
        success: false,
        message: 'Error exporting report data'
      });
    }
  }
};

// Helper functions for custom reports
async function generateFinancialSummary(filters) {
  // Implementation for financial summary
  return { summary: 'Financial summary data' };
}

async function generatePropertyPerformance(filters) {
  // Implementation for property performance
  return { summary: 'Property performance data' };
}

async function generateTenantActivity(filters) {
  // Implementation for tenant activity
  return { summary: 'Tenant activity data' };
}

module.exports = reportController;