const { Payment, Lease, Property, User } = require('../models');
const { Op } = require('sequelize');
const moment = require('moment');
const paymentProcessor = require('../utils/paymentProcessor');

const paymentController = {
  // Get all payments
  getAllPayments: async (req, res) => {
    try {
      const { page = 1, limit = 10, status, leaseId, propertyId, tenantId, landlordId } = req.query;
      const offset = (page - 1) * limit;

      const where = {};
      if (status) where.status = status;
      if (leaseId) where.leaseId = leaseId;
      if (propertyId) where.propertyId = propertyId;
      if (tenantId) where.tenantId = tenantId;
      if (landlordId) where.landlordId = landlordId;

      // Filter based on user role
      if (req.user.role === 'tenant') {
        where.tenantId = req.user.id;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      const payments = await Payment.findAndCountAll({
        where,
        include: [
          {
            model: Lease,
            as: 'lease',
            attributes: ['id', 'startDate', 'endDate']
          },
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email']
          },
          {
            model: User,
            as: 'landlord',
            attributes: ['id', 'firstName', 'lastName', 'email']
          }
        ],
        limit: parseInt(limit),
        offset: parseInt(offset),
        order: [['dueDate', 'DESC']]
      });

      res.json({
        success: true,
        data: {
          payments: payments.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: payments.count,
            pages: Math.ceil(payments.count / limit)
          }
        }
      });
    } catch (error) {
      console.error('Get payments error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching payments'
      });
    }
  },

  // Get payment by ID
  getPaymentById: async (req, res) => {
    try {
      const { id } = req.params;

      const payment = await Payment.findByPk(id, {
        include: [
          {
            model: Lease,
            as: 'lease',
            include: ['property', 'tenant']
          },
          {
            model: Property,
            as: 'property'
          },
          {
            model: User,
            as: 'tenant',
            attributes: { exclude: ['password'] }
          },
          {
            model: User,
            as: 'landlord',
            attributes: { exclude: ['password'] }
          }
        ]
      });

      if (!payment) {
        return res.status(404).json({
          success: false,
          message: 'Payment not found'
        });
      }

      // Check permissions
      if (payment.tenantId !== req.user.id && payment.landlordId !== req.user.id && !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to view this payment'
        });
      }

      res.json({
        success: true,
        data: { payment }
      });
    } catch (error) {
      console.error('Get payment error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching payment'
      });
    }
  },

  // Create payment (manual entry)
  createPayment: async (req, res) => {
    try {
      const paymentData = {
        ...req.body,
        landlordId: req.user.role === 'landlord' ? req.user.id : req.body.landlordId
      };

      // Verify lease exists
      const lease = await Lease.findByPk(paymentData.leaseId);
      if (!lease) {
        return res.status(404).json({
          success: false,
          message: 'Lease not found'
        });
      }

      const payment = await Payment.create(paymentData);

      const createdPayment = await Payment.findByPk(payment.id, {
        include: ['lease', 'property', 'tenant', 'landlord']
      });

      res.status(201).json({
        success: true,
        message: 'Payment created successfully',
        data: { payment: createdPayment }
      });
    } catch (error) {
      console.error('Create payment error:', error);
      res.status(500).json({
        success: false,
        message: 'Error creating payment'
      });
    }
  },

  // Process payment (tenant making payment)
  processPayment: async (req, res) => {
    try {
      const { id } = req.params;
      const { paymentMethod, paymentData } = req.body;

      const payment = await Payment.findByPk(id);
      if (!payment) {
        return res.status(404).json({
          success: false,
          message: 'Payment not found'
        });
      }

      // Check if payment belongs to the tenant
      if (payment.tenantId !== req.user.id) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to process this payment'
        });
      }

      // Check if payment is already paid
      if (payment.status === 'paid') {
        return res.status(400).json({
          success: false,
          message: 'Payment has already been processed'
        });
      }

      // Process payment through payment processor
      const paymentResult = await paymentProcessor.processPayment({
        amount: payment.amount,
        currency: 'USD',
        paymentMethod,
        ...paymentData,
        description: `Rent payment for ${payment.property.address}`,
        metadata: {
          paymentId: payment.id,
          leaseId: payment.leaseId,
          propertyId: payment.propertyId,
          tenantId: payment.tenantId
        }
      });

      if (!paymentResult.success) {
        return res.status(400).json({
          success: false,
          message: `Payment failed: ${paymentResult.error}`
        });
      }

      // Update payment record
      await payment.update({
        status: 'paid',
        paidDate: new Date(),
        paymentMethod,
        transactionId: paymentResult.transactionId,
        receiptUrl: paymentResult.receiptUrl
      });

      const updatedPayment = await Payment.findByPk(id, {
        include: ['lease', 'property', 'tenant', 'landlord']
      });

      res.json({
        success: true,
        message: 'Payment processed successfully',
        data: { payment: updatedPayment }
      });
    } catch (error) {
      console.error('Process payment error:', error);
      res.status(500).json({
        success: false,
        message: 'Error processing payment'
      });
    }
  },

  // Update payment
  updatePayment: async (req, res) => {
    try {
      const { id } = req.params;
      const updateData = req.body;

      const payment = await Payment.findByPk(id);
      if (!payment) {
        return res.status(404).json({
          success: false,
          message: 'Payment not found'
        });
      }

      // Check permissions
      if (payment.landlordId !== req.user.id && !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to update this payment'
        });
      }

      await payment.update(updateData);

      const updatedPayment = await Payment.findByPk(id, {
        include: ['lease', 'property', 'tenant', 'landlord']
      });

      res.json({
        success: true,
        message: 'Payment updated successfully',
        data: { payment: updatedPayment }
      });
    } catch (error) {
      console.error('Update payment error:', error);
      res.status(500).json({
        success: false,
        message: 'Error updating payment'
      });
    }
  },

  // Record manual payment
  recordManualPayment: async (req, res) => {
    try {
      const { id } = req.params;
      const { paymentMethod, referenceNumber, notes } = req.body;

      const payment = await Payment.findByPk(id);
      if (!payment) {
        return res.status(404).json({
          success: false,
          message: 'Payment not found'
        });
      }

      // Check permissions
      if (payment.landlordId !== req.user.id && !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to record manual payment'
        });
      }

      await payment.update({
        status: 'paid',
        paidDate: new Date(),
        paymentMethod: paymentMethod || 'manual',
        transactionId: referenceNumber,
        description: notes ? `${payment.description} - ${notes}` : payment.description
      });

      res.json({
        success: true,
        message: 'Manual payment recorded successfully',
        data: { payment }
      });
    } catch (error) {
      console.error('Record manual payment error:', error);
      res.status(500).json({
        success: false,
        message: 'Error recording manual payment'
      });
    }
  },

  // Refund payment
  refundPayment: async (req, res) => {
    try {
      const { id } = req.params;
      const { refundAmount, reason } = req.body;

      const payment = await Payment.findByPk(id);
      if (!payment) {
        return res.status(404).json({
          success: false,
          message: 'Payment not found'
        });
      }

      if (payment.status !== 'paid') {
        return res.status(400).json({
          success: false,
          message: 'Only paid payments can be refunded'
        });
      }

      // Check permissions
      if (payment.landlordId !== req.user.id && !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to refund this payment'
        });
      }

      // Process refund through payment processor
      const refundResult = await paymentProcessor.refundPayment(
        payment.transactionId,
        refundAmount || payment.amount,
        reason
      );

      if (!refundResult.success) {
        return res.status(400).json({
          success: false,
          message: `Refund failed: ${refundResult.error}`
        });
      }

      // Update payment record
      await payment.update({
        status: 'refunded',
        description: reason ? `${payment.description} - Refunded: ${reason}` : `${payment.description} - Refunded`
      });

      res.json({
        success: true,
        message: 'Payment refunded successfully',
        data: { payment }
      });
    } catch (error) {
      console.error('Refund payment error:', error);
      res.status(500).json({
        success: false,
        message: 'Error refunding payment'
      });
    }
  },

  // Get overdue payments
  getOverduePayments: async (req, res) => {
    try {
      const { landlordId } = req.query;
      const where = {
        status: 'pending',
        dueDate: { [Op.lt]: new Date() }
      };

      if (landlordId) {
        where.landlordId = landlordId;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      const overduePayments = await Payment.findAll({
        where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
          }
        ],
        order: [['dueDate', 'ASC']]
      });

      res.json({
        success: true,
        data: { payments: overduePayments }
      });
    } catch (error) {
      console.error('Get overdue payments error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching overdue payments'
      });
    }
  },

  // Generate financial report
  generateFinancialReport: async (req, res) => {
    try {
      const { startDate, endDate, landlordId } = req.query;
      const where = {};

      if (landlordId) {
        where.landlordId = landlordId;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      if (startDate && endDate) {
        where.paidDate = {
          [Op.between]: [new Date(startDate), new Date(endDate)]
        };
      }

      const payments = await Payment.findAll({
        where: { ...where, status: 'paid' },
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName']
          }
        ],
        order: [['paidDate', 'DESC']]
      });

      const totalRevenue = payments.reduce((sum, payment) => sum + parseFloat(payment.amount), 0);

      res.json({
        success: true,
        data: {
          payments,
          summary: {
            totalRevenue,
            totalPayments: payments.length,
            period: {
              startDate: startDate || 'Beginning',
              endDate: endDate || 'Present'
            }
          }
        }
      });
    } catch (error) {
      console.error('Generate financial report error:', error);
      res.status(500).json({
        success: false,
        message: 'Error generating financial report'
      });
    }
  }
};

module.exports = paymentController;