const { Maintenance, Property, User, AuditLog } = require('../models');
const { Op } = require('sequelize');
const moment = require('moment');

const maintenanceController = {
  // Get all maintenance requests
  getAllMaintenance: async (req, res) => {
    try {
      const { page = 1, limit = 10, status, priority, category, propertyId, tenantId } = req.query;
      const offset = (page - 1) * limit;

      const where = {};
      if (status) where.status = status;
      if (priority) where.priority = priority;
      if (category) where.category = category;
      if (propertyId) where.propertyId = propertyId;
      if (tenantId) where.tenantId = tenantId;

      // Filter based on user role
      if (req.user.role === 'tenant') {
        where.tenantId = req.user.id;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      } else if (req.user.role.includes('maintenance')) {
        where.assignedTo = req.user.id;
      }

      const maintenanceRequests = await Maintenance.findAndCountAll({
        where,
        include: [
          {
            model: Property,
            as: 'property',
            attributes: ['id', 'title', 'address', 'images']
          },
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
          },
          {
            model: User,
            as: 'assignedStaff',
            attributes: ['id', 'firstName', 'lastName', 'email', 'phone']
          }
        ],
        limit: parseInt(limit),
        offset: parseInt(offset),
        order: [['createdAt', 'DESC']]
      });

      res.json({
        success: true,
        data: {
          maintenanceRequests: maintenanceRequests.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: maintenanceRequests.count,
            pages: Math.ceil(maintenanceRequests.count / limit)
          }
        }
      });
    } catch (error) {
      console.error('Get maintenance requests error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching maintenance requests'
      });
    }
  },

  // Get maintenance by ID
  getMaintenanceById: async (req, res) => {
    try {
      const { id } = req.params;

      const maintenance = await Maintenance.findByPk(id, {
        include: [
          {
            model: Property,
            as: 'property',
            include: ['landlord']
          },
          {
            model: User,
            as: 'tenant',
            attributes: { exclude: ['password'] }
          },
          {
            model: User,
            as: 'assignedStaff',
            attributes: { exclude: ['password'] }
          },
          {
            model: User,
            as: 'landlord',
            attributes: { exclude: ['password'] }
          }
        ]
      });

      if (!maintenance) {
        return res.status(404).json({
          success: false,
          message: 'Maintenance request not found'
        });
      }

      // Check permissions
      if (maintenance.tenantId !== req.user.id && 
          maintenance.landlordId !== req.user.id && 
          maintenance.assignedTo !== req.user.id && 
          !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to view this maintenance request'
        });
      }

      res.json({
        success: true,
        data: { maintenance }
      });
    } catch (error) {
      console.error('Get maintenance error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching maintenance request'
      });
    }
  },

  // Create maintenance request
  createMaintenance: async (req, res) => {
    try {
      const maintenanceData = {
        ...req.body,
        tenantId: req.user.id,
        images: req.uploadedFiles || []
      };

      // Verify property exists and belongs to tenant
      const property = await Property.findByPk(maintenanceData.propertyId);
      if (!property) {
        return res.status(404).json({
          success: false,
          message: 'Property not found'
        });
      }

      // Set landlord ID from property
      maintenanceData.landlordId = property.landlordId;

      const maintenance = await Maintenance.create(maintenanceData);

      const createdMaintenance = await Maintenance.findByPk(maintenance.id, {
        include: ['property', 'tenant', 'landlord']
      });

      res.status(201).json({
        success: true,
        message: 'Maintenance request created successfully',
        data: { maintenance: createdMaintenance }
      });
    } catch (error) {
      console.error('Create maintenance error:', error);
      res.status(500).json({
        success: false,
        message: 'Error creating maintenance request'
      });
    }
  },

  // Update maintenance request
  updateMaintenance: async (req, res) => {
    try {
      const { id } = req.params;
      const updateData = req.body;

      const maintenance = await Maintenance.findByPk(id);
      if (!maintenance) {
        return res.status(404).json({
          success: false,
          message: 'Maintenance request not found'
        });
      }

      // Check permissions
      if (maintenance.tenantId !== req.user.id && 
          maintenance.landlordId !== req.user.id && 
          !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to update this maintenance request'
        });
      }

      // Add new images if uploaded
      if (req.uploadedFiles && req.uploadedFiles.length > 0) {
        const existingImages = maintenance.images || [];
        updateData.images = [...existingImages, ...req.uploadedFiles];
      }

      await maintenance.update(updateData);

      const updatedMaintenance = await Maintenance.findByPk(id, {
        include: ['property', 'tenant', 'assignedStaff', 'landlord']
      });

      res.json({
        success: true,
        message: 'Maintenance request updated successfully',
        data: { maintenance: updatedMaintenance }
      });
    } catch (error) {
      console.error('Update maintenance error:', error);
      res.status(500).json({
        success: false,
        message: 'Error updating maintenance request'
      });
    }
  },

  // Assign maintenance staff
  assignMaintenance: async (req, res) => {
    try {
      const { id } = req.params;
      const { assignedTo } = req.body;

      const maintenance = await Maintenance.findByPk(id);
      if (!maintenance) {
        return res.status(404).json({
          success: false,
          message: 'Maintenance request not found'
        });
      }

      // Verify assigned staff exists and has maintenance role
      const staff = await User.findByPk(assignedTo);
      if (!staff || !staff.role.includes('maintenance')) {
        return res.status(400).json({
          success: false,
          message: 'Invalid maintenance staff member'
        });
      }

      await maintenance.update({
        assignedTo,
        status: 'in_progress'
      });

      const updatedMaintenance = await Maintenance.findByPk(id, {
        include: ['property', 'tenant', 'assignedStaff', 'landlord']
      });

      res.json({
        success: true,
        message: 'Maintenance staff assigned successfully',
        data: { maintenance: updatedMaintenance }
      });
    } catch (error) {
      console.error('Assign maintenance error:', error);
      res.status(500).json({
        success: false,
        message: 'Error assigning maintenance staff'
      });
    }
  },

  // Update maintenance status
  updateStatus: async (req, res) => {
    try {
      const { id } = req.params;
      const { status } = req.body;

      const maintenance = await Maintenance.findByPk(id);
      if (!maintenance) {
        return res.status(404).json({
          success: false,
          message: 'Maintenance request not found'
        });
      }

      // Check permissions
      if (maintenance.assignedTo !== req.user.id && 
          maintenance.landlordId !== req.user.id && 
          !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to update this maintenance status'
        });
      }

      const updateData = { status };
      
      // Set completed date if status is completed
      if (status === 'completed') {
        updateData.completedDate = new Date();
      }

      await maintenance.update(updateData);

      res.json({
        success: true,
        message: `Maintenance status updated to ${status}`,
        data: { maintenance }
      });
    } catch (error) {
      console.error('Update maintenance status error:', error);
      res.status(500).json({
        success: false,
        message: 'Error updating maintenance status'
      });
    }
  },

  // Add maintenance notes
  addNotes: async (req, res) => {
    try {
      const { id } = req.params;
      const { notes } = req.body;

      const maintenance = await Maintenance.findByPk(id);
      if (!maintenance) {
        return res.status(404).json({
          success: false,
          message: 'Maintenance request not found'
        });
      }

      // Check permissions
      if (maintenance.assignedTo !== req.user.id && 
          maintenance.landlordId !== req.user.id && 
          !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to add notes to this maintenance request'
        });
      }

      const existingNotes = maintenance.notes || '';
      const updatedNotes = existingNotes ? `${existingNotes}\n${new Date().toLocaleString()}: ${notes}` : `${new Date().toLocaleString()}: ${notes}`;

      await maintenance.update({ notes: updatedNotes });

      res.json({
        success: true,
        message: 'Notes added successfully',
        data: { maintenance }
      });
    } catch (error) {
      console.error('Add maintenance notes error:', error);
      res.status(500).json({
        success: false,
        message: 'Error adding maintenance notes'
      });
    }
  },

  // Complete maintenance
  completeMaintenance: async (req, res) => {
    try {
      const { id } = req.params;
      const { cost, finalNotes } = req.body;

      const maintenance = await Maintenance.findByPk(id);
      if (!maintenance) {
        return res.status(404).json({
          success: false,
          message: 'Maintenance request not found'
        });
      }

      // Check permissions
      if (maintenance.assignedTo !== req.user.id && 
          maintenance.landlordId !== req.user.id && 
          !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to complete this maintenance request'
        });
      }

      const updateData = {
        status: 'completed',
        completedDate: new Date(),
        cost: cost || maintenance.cost
      };

      if (finalNotes) {
        const existingNotes = maintenance.notes || '';
        updateData.notes = existingNotes ? `${existingNotes}\nCompletion: ${finalNotes}` : `Completion: ${finalNotes}`;
      }

      await maintenance.update(updateData);

      const updatedMaintenance = await Maintenance.findByPk(id, {
        include: ['property', 'tenant', 'assignedStaff', 'landlord']
      });

      res.json({
        success: true,
        message: 'Maintenance request completed successfully',
        data: { maintenance: updatedMaintenance }
      });
    } catch (error) {
      console.error('Complete maintenance error:', error);
      res.status(500).json({
        success: false,
        message: 'Error completing maintenance request'
      });
    }
  },

  // Cancel maintenance
  cancelMaintenance: async (req, res) => {
    try {
      const { id } = req.params;
      const { reason } = req.body;

      const maintenance = await Maintenance.findByPk(id);
      if (!maintenance) {
        return res.status(404).json({
          success: false,
          message: 'Maintenance request not found'
        });
      }

      // Check permissions
      if (maintenance.tenantId !== req.user.id && 
          maintenance.landlordId !== req.user.id && 
          !req.user.role.includes('admin')) {
        return res.status(403).json({
          success: false,
          message: 'Not authorized to cancel this maintenance request'
        });
      }

      await maintenance.update({
        status: 'cancelled',
        notes: reason ? `${maintenance.notes || ''}\nCancelled: ${reason}` : maintenance.notes
      });

      res.json({
        success: true,
        message: 'Maintenance request cancelled successfully',
        data: { maintenance }
      });
    } catch (error) {
      console.error('Cancel maintenance error:', error);
      res.status(500).json({
        success: false,
        message: 'Error cancelling maintenance request'
      });
    }
  },

  // Get maintenance by property
  getMaintenanceByProperty: async (req, res) => {
    try {
      const { propertyId } = req.params;
      const { page = 1, limit = 10, status } = req.query;
      const offset = (page - 1) * limit;

      const where = { propertyId };
      if (status) where.status = status;

      const maintenanceRequests = await Maintenance.findAndCountAll({
        where,
        include: [
          {
            model: User,
            as: 'tenant',
            attributes: ['id', 'firstName', 'lastName', 'email']
          },
          {
            model: User,
            as: 'assignedStaff',
            attributes: ['id', 'firstName', 'lastName', 'email']
          }
        ],
        limit: parseInt(limit),
        offset: parseInt(offset),
        order: [['createdAt', 'DESC']]
      });

      res.json({
        success: true,
        data: {
          maintenanceRequests: maintenanceRequests.rows,
          pagination: {
            page: parseInt(page),
            limit: parseInt(limit),
            total: maintenanceRequests.count,
            pages: Math.ceil(maintenanceRequests.count / limit)
          }
        }
      });
    } catch (error) {
      console.error('Get maintenance by property error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching maintenance requests for property'
      });
    }
  },

  // Get maintenance statistics
  getMaintenanceStats: async (req, res) => {
    try {
      const { landlordId } = req.query;
      const where = {};

      if (landlordId) {
        where.landlordId = landlordId;
      } else if (req.user.role === 'landlord') {
        where.landlordId = req.user.id;
      }

      const totalRequests = await Maintenance.count({ where });
      const pendingRequests = await Maintenance.count({ where: { ...where, status: 'pending' } });
      const inProgressRequests = await Maintenance.count({ where: { ...where, status: 'in_progress' } });
      const completedRequests = await Maintenance.count({ where: { ...where, status: 'completed' } });

      // Average completion time
      const completedMaintenance = await Maintenance.findAll({
        where: { ...where, status: 'completed', completedDate: { [Op.not]: null } },
        attributes: ['createdAt', 'completedDate']
      });

      let avgCompletionTime = 0;
      if (completedMaintenance.length > 0) {
        const totalTime = completedMaintenance.reduce((sum, maintenance) => {
          const completionTime = new Date(maintenance.completedDate) - new Date(maintenance.createdAt);
          return sum + completionTime;
        }, 0);
        avgCompletionTime = totalTime / completedMaintenance.length;
      }

      // Requests by priority
      const priorityStats = await Maintenance.findAll({
        where,
        attributes: [
          'priority',
          [Maintenance.sequelize.fn('COUNT', Maintenance.sequelize.col('id')), 'count']
        ],
        group: ['priority']
      });

      // Requests by category
      const categoryStats = await Maintenance.findAll({
        where,
        attributes: [
          'category',
          [Maintenance.sequelize.fn('COUNT', Maintenance.sequelize.col('id')), 'count']
        ],
        group: ['category']
      });

      res.json({
        success: true,
        data: {
          stats: {
            totalRequests,
            pendingRequests,
            inProgressRequests,
            completedRequests,
            avgCompletionTime: Math.round(avgCompletionTime / (1000 * 60 * 60 * 24)) // Convert to days
          },
          priorityStats,
          categoryStats
        }
      });
    } catch (error) {
      console.error('Get maintenance stats error:', error);
      res.status(500).json({
        success: false,
        message: 'Error fetching maintenance statistics'
      });
    }
  }
};

module.exports = maintenanceController;