const bcrypt = require('bcryptjs');
const jwt = require('jsonwebtoken');
const db = require('../config/database');

const authController = {
  // Register new user
  register: async (req, res) => {
    try {
      const { firstName, lastName, email, password, role = 'tenant' } = req.body;

      console.log('Registration attempt:', { firstName, lastName, email, role });

      // Check if user exists
      const [existingUsers] = await db.execute(
        'SELECT id FROM users WHERE email = ?',
        [email]
      );

      if (existingUsers.length > 0) {
        return res.status(400).json({
          success: false,
          message: 'User already exists with this email'
        });
      }

      // Hash password
      const hashedPassword = await bcrypt.hash(password, 12);

      // Create user
      const [result] = await db.execute(
        'INSERT INTO users (first_name, last_name, email, password_hash, role_code) VALUES (?, ?, ?, ?, ?)',
        [firstName, lastName, email, hashedPassword, role]
      );

      console.log('User created with ID:', result.insertId);

      // Generate token
      const token = jwt.sign(
        { userId: result.insertId, email, role },
        process.env.JWT_SECRET || 'fallback-secret',
        { expiresIn: process.env.JWT_EXPIRES_IN || '24h' }
      );

      res.status(201).json({
        success: true,
        message: 'User registered successfully',
        token,
        user: { id: result.insertId, firstName, lastName, email, role }
      });
    } catch (error) {
      console.error('Register error:', error);
      res.status(500).json({
        success: false,
        message: 'Registration failed',
        error: error.message
      });
    }
  },

  // Login user
  login: async (req, res) => {
    try {
      const { email, password } = req.body;

      // Find user
      const [users] = await db.execute(
        'SELECT * FROM users WHERE email = ?',
        [email]
      );

      if (users.length === 0) {
        return res.status(401).json({
          success: false,
          message: 'Invalid credentials'
        });
      }

      const user = users[0];

      // Check password
      const isValidPassword = await bcrypt.compare(password, user.password_hash);
      if (!isValidPassword) {
        return res.status(401).json({
          success: false,
          message: 'Invalid credentials'
        });
      }

      // Generate token
      const token = jwt.sign(
        { userId: user.id, email: user.email, role: user.role_code },
        process.env.JWT_SECRET,
        { expiresIn: process.env.JWT_EXPIRES_IN }
      );

      res.json({
        success: true,
        message: 'Login successful',
        token,
        user: {
          id: user.id,
          firstName: user.first_name,
          lastName: user.last_name,
          email: user.email,
          role: user.role_code
        }
      });
    } catch (error) {
      console.error('Login error:', error);
      res.status(500).json({
        success: false,
        message: 'Login failed'
      });
    }
  },

  // Get user profile
  getProfile: async (req, res) => {
    try {
      const [users] = await db.execute(
        'SELECT id, first_name, last_name, email, role_code FROM users WHERE id = ?',
        [req.user.userId]
      );

      if (users.length === 0) {
        return res.status(404).json({
          success: false,
          message: 'User not found'
        });
      }

      const user = users[0];
      res.json({
        success: true,
        user: {
          id: user.id,
          firstName: user.first_name,
          lastName: user.last_name,
          email: user.email,
          role: user.role_code
        }
      });
    } catch (error) {
      console.error('Get profile error:', error);
      res.status(500).json({
        success: false,
        message: 'Failed to get profile'
      });
    }
  },

  // Placeholder methods
  refreshToken: (req, res) => {
    res.status(501).json({ success: false, message: 'Not implemented' });
  },

  forgotPassword: (req, res) => {
    res.status(501).json({ success: false, message: 'Not implemented' });
  },

  resetPassword: (req, res) => {
    res.status(501).json({ success: false, message: 'Not implemented' });
  },

  updateProfile: (req, res) => {
    res.status(501).json({ success: false, message: 'Not implemented' });
  },

  changePassword: (req, res) => {
    res.status(501).json({ success: false, message: 'Not implemented' });
  },

  logout: (req, res) => {
    res.json({ success: true, message: 'Logged out successfully' });
  }
};

module.exports = authController;