const express = require('express');
const cors = require('cors');
const morgan = require('morgan');
const { notFound, errorHandler } = require('./middleware/errorHandler');
const { apiLimiter, authLimiter, searchLimiter, helmetConfig, sanitizeInput } = require('./middleware/security');

// Import routes
// Import routes with error handling
let authRoutes, propertyRoutes, dashboardRoutes, searchRoutes;
try {
  authRoutes = require('./routes/auth');
  propertyRoutes = require('./routes/properties');
  dashboardRoutes = require('./routes/dashboard');
  searchRoutes = require('./routes/search');
} catch (error) {
  console.error('Route import error:', error.message);
}
// Add other routes as needed

const app = express();

// Security middleware
app.use(helmetConfig);
app.use(sanitizeInput);

// CORS configuration
app.use(cors({
  origin: ['http://localhost:3000', 'http://localhost:5173'],
  credentials: true,
  methods: ['GET', 'POST', 'PUT', 'DELETE', 'OPTIONS'],
  allowedHeaders: ['Content-Type', 'Authorization']
}));

// Rate limiting (disabled for debugging)
// app.use('/api/', apiLimiter);
// app.use('/api/auth', authLimiter);
// app.use('/api/search', searchLimiter);

// Body parsing middleware
app.use(express.json({ limit: '10mb' }));
app.use(express.urlencoded({ extended: true }));

// Debug middleware
app.use((req, res, next) => {
  console.log(`${req.method} ${req.path}`, req.body);
  next();
});

// Logging
if (process.env.NODE_ENV === 'development') {
  app.use(morgan('dev'));
} else {
  app.use(morgan('combined'));
}

// Health check route
app.get('/health', (req, res) => {
  res.status(200).json({
    success: true,
    message: 'StaySpot API is running',
    timestamp: new Date().toISOString(),
    environment: process.env.NODE_ENV
  });
});

// API routes with error handling
if (authRoutes) app.use('/api/auth', authRoutes);
if (propertyRoutes) app.use('/api/properties', propertyRoutes);
if (dashboardRoutes) app.use('/api/dashboard', dashboardRoutes);
if (searchRoutes) app.use('/api/search', searchRoutes);

// Optional routes
try {
  app.use('/api/partnerships', require('./routes/partnerships'));
} catch (e) { console.log('Partnerships route not available'); }

try {
  app.use('/api/chat', require('./routes/chat'));
} catch (e) { console.log('Chat route not available'); }

try {
  app.use('/api/transportation', require('./routes/transportation'));
} catch (e) { console.log('Transportation route not available'); }

try {
  app.use('/api/analytics', require('./routes/analytics'));
} catch (e) { console.log('Analytics route not available'); }
// Add other API routes

// Serve uploaded files statically
app.use('/uploads', express.static('uploads'));

// 404 handler
app.use(notFound);

// Error handler
app.use(errorHandler);

module.exports = app;